/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/export.h>
#include <agxSensor/TriaxialCrossSensitivity.h>
#include <agxSensor/TriaxialRange.h>
#include <agxSensor/TriaxialSignalSystemNode.h>

#include <agxStream/Serializable.h>

#include <agx/Referenced.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( MagnetometerModel );

  /**
  Base magnetometer model describing the fundamental parameters, such as measurement range and
  zero flux bias, of an magnetometer.
  \note The configuration in this model is shared and synchronized across all instances.
  */
  class AGXSENSOR_EXPORT MagnetometerModel : public agx::Referenced, public agxStream::Serializable
  {
    public:
      /**
      Creates a magnetometer model outputting ideal measurement data (ground truth values).
      \return new ideal magnetometer model
      */
      static MagnetometerModel* makeIdealModel();

    public:
      /**
      Constructs a magnetometer model with the specified parameters.
      \param range - detectable range of magnetic flux density, in T
      \param crossSensitivity - cross axis sensitivity
      \param zeroFluxBias - bias magnetic flux density reported in each axis under conditions
                            without externally applied field, in T
      \param outputModifiers - list of modifiers to apply to the output of the magnetometer
      */
      MagnetometerModel( TriaxialRange range = agx::RangeReal(
                            std::numeric_limits<agx::Real>::lowest(),
                            std::numeric_limits<agx::Real>::max() ),
                         TriaxialCrossSensitivity crossSensitivity = 0.01,
                         agx::Vec3 zeroFluxBias = agx::Vec3( 0.0 ),
                         ITriaxialSignalSystemNodeRefVector outputModifiers = ITriaxialSignalSystemNodeRefVector() );

      /**
      Set the detectable range for the magnetometer.
      \param range - detectable range of magnetic flux density, in T
      */
      void setRange( const TriaxialRange& range );

      /**
      \return the range of measurable magnetic flux density, in T, that the magnetometer can detect.
      */
      const TriaxialRange& getRange() const;

      /**
      Set the magnetometer cross-axis sensitivity.
      \param crossSensitivity - cross axis sensitivity
      */
      void setCrossAxisSensitivity( const TriaxialCrossSensitivity& crossSensitivity );

      /**
      \return the cross sensitivity, describing the effects that magnetic flux density along one
              axis has on every other axis.
      */
      const TriaxialCrossSensitivity& getCrossAxisSensitivity() const;

      /**
      Set the bias in magnetic flux density under conditions without externally applied magnetic
      field.
      \param zeroFluxBias - bias magnetic flux density reported in each axis under conditions
                            without externally applied field, in T
      */
      void setZeroFluxBias( const agx::Vec3& zeroFluxBias );

      /**
      \return the bias magnetic flux density, in T, reported in each axis under conditions without
              externally applied magnetic field.
      */
      const agx::Vec3& getZeroFluxBias() const;

      /**
      \return the list of output modifiers to apply to the output
      */
      const ITriaxialSignalSystemNodeRefVector& getOutputModifiers() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~MagnetometerModel() = default;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::MagnetometerModel );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      TriaxialRange m_range;
      TriaxialCrossSensitivity m_crossAxisSensitivity;
      agx::Vec3 m_zeroFluxBias;
      ITriaxialSignalSystemNodeRefVector m_outputModifiers;
  };
}
