/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <memory>
#include <openplx/Physics/Signals/InputSignal.h>

namespace openplx::Physics::Signals {
    class Output;
    class Input;
}

namespace agxopenplx {
    class OutputSignalQueue;
    class SignalSourceMapper;

    /**
     * Takes a OpenPLX input signal and updates the AGX simulation.
     * Looks up the AGX objects from a SignalSourceMapper.
     */
    class InputHandler {
        public:
            InputHandler(
                std::shared_ptr<SignalSourceMapper> mapper);

            template<class InputType>
            bool handle(
                const std::shared_ptr<openplx::Physics::Signals::InputSignal>& input_signal,
                const std::shared_ptr<InputType>& input);

            template<class InputType>
            bool handleIfMatch(
                const std::shared_ptr<openplx::Physics::Signals::InputSignal>& input_signal) {
                if (input_signal == nullptr) return false;
                auto target = input_signal->target();
                if (target == nullptr) return false;
                if (auto matched_input = std::dynamic_pointer_cast<InputType>(target)) {
                    return handle<InputType>(input_signal, matched_input);
                }
                return false;
            }
        private:
            std::shared_ptr<SignalSourceMapper> m_mapper;
    };

    /**
     * Takes a OpenPLX output and queues a openplx output signal by
     * reading state from the AGX simulation.
     * Looks up the AGX objects from a SignalSourceMapper.
     */
    class OutputHandler {
        public:
            OutputHandler(
                std::shared_ptr<OutputSignalQueue> output_queue,
                std::shared_ptr<SignalSourceMapper> mapper);

            template<class OutputType>
            bool handle(const std::shared_ptr<OutputType>& output);

            template<class OutputType>
            bool handleIfMatch(const std::shared_ptr<openplx::Physics::Signals::Output>& output) {
                if (auto matched_output = std::dynamic_pointer_cast<OutputType>(output)) {
                    return handle<OutputType>(matched_output);
                }
                return false;
            }
        private:
            std::shared_ptr<OutputSignalQueue> m_output_queue;
            std::shared_ptr<SignalSourceMapper> m_mapper;
    };
}
