/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxOSG/export.h>
#include <agx/Math.h>
#include <agx/Referenced.h>
#include <agx/RigidBody.h>
#include <agxOSG/GeometryNode.h>


namespace agxOSG
{
  AGX_DECLARE_POINTER_TYPES(RigidBodyRenderCache);
  /**
  * RigidBodyRenderCache is used to cache generated visuals and lookup mappings of
  * RigidBodies cloned from Templates in agx::RigidBodyEmitter and other related
  * granular applications. It is used to optimize rendering by basic instantiation
  * of similar template bodies and also for fast lookup to circumvent the complexity
  * of agxOSG::findGeometryNode.
  */
  class AGXOSG_EXPORT RigidBodyRenderCache : public agx::Referenced
  {
  public:
    AGX_DECLARE_POINTER_TYPES(RigidBodyVisualGroup);
    /**
    * A help class to store all information we need to copy a rigid body visualization to duplicate the rigid body from 
    * a rigid body template.
    */
    class RigidBodyVisualGroup : public agx::Referenced
    {
      public:

        RigidBodyVisualGroup() :
          m_group(nullptr) { }

        RigidBodyVisualGroup(osg::Group* group, std::vector<size_t> geometryIndex) :
          m_group(group), m_geometryIndices(geometryIndex) { }

        agxCollide::GeometryRef getGeometry(agx::RigidBody* newBody, size_t visualIndex) const;

        bool isValid() const { return m_group != nullptr; }

        size_t getNumChildren() const { return isValid() ? m_group->getNumChildren() : 0; }

        osg::Group* getGroup() const { return m_group; }

      private:
        osg::ref_ptr<osg::Group> m_group;
        // a list with indices linking each visual node (child in m_group) to a geometry in the rigid body
        std::vector<size_t> m_geometryIndices;
    };

  public:
    /**
    * Default constructor.
    * \param onlyAddEmittedBodies - True if only emitted RigidBodies
                                    should be added, false otherwise.
    */
    RigidBodyRenderCache(bool onlyAddEmittedBodies=true);

    /**
    Add the osg::Group created from a RigidBody to the cache.
    \param rootNode - the rootNode that is used to build the cache.
    */
    void addToCache(osg::Group* rootNode);

    /**
    Get the current mapping in the cache between a RigidBody to a osg::Group.
    \return the current osg::Group mapped from the RigidBody.
    */
    osg::Group* getNode(agx::RigidBody* body);

    /**
    \return true if only emitted bodies should be added, false otherwise.
    */
    bool getOnlyAddEmittedBodies() const;

    /**
    \param enable - true if only emitted bodies should be added, false otherwise.
    */
    void setOnlyAddEmittedBodies( bool enable );

    /**
    * Add template body to visual cache that is used during playback in qtViewer.
    * \param body - The specified template body to add to node cache.
    * \return the cached visual node, with all properties we need to recreated it, for the template body.
              An empty group if the body isn't a template.
    */
    const RigidBodyRenderCache::RigidBodyVisualGroup& addTemplateBodyToVisualCache( agx::RigidBody* body );

    /**
    * Get cached template body visual from the node cache if it exists.
    * \return the cached visual node, with all properties we need to recreated it, for the template body if it exists,
              otherwise an empty group.
    */
    const RigidBodyRenderCache::RigidBodyVisualGroup& getCachedTemplateBodyVisual( agx::RigidBody* body );

    /**
    * Gets cached template body visual from the node cache if it exists,
    * otherwise a new visual node will be created in the cache and returned.
    * If the specified body is not a generated from a Template, the function
    * returns nullptr.
    * \param body - the specified body that should be used to get or create
    *               the visual representation from the node cache.
    * \return the cached visual node for the template body. This is a an empty group if the RB isn't template.
    */
    const RigidBodyRenderCache::RigidBodyVisualGroup& getOrCreateCachedTemplateBodyNode( agx::RigidBody* body );

    /**
    * Checks wherever the specifed body is generated/cloned from the temlpate
    * body via RigidBodyEmitter och creation via the Momentum API.
    * \note - the function checks if a string property called "TemplateUUID"
    *         eixsts in the body's property container.
    */
    bool bodyIsTemplateEmitted( agx::RigidBody* body );

    /**
    * Clear the internal cache.
    */
    void clear();

  protected:
    virtual ~RigidBodyRenderCache();
    void traverseAndInsert(osg::Group* rootNode);
    bool shouldAddBody(agx::RigidBody* body) const;

  private:
    typedef agx::HashTable<agx::RigidBody*, osg::Group*> BodyToGroupNodeTable;
    typedef agx::HashTable<agx::String, RigidBodyRenderCache::RigidBodyVisualGroupRef> BodyToRenderGroupTable;

    BodyToGroupNodeTable   m_bodiesToGroupNodes;
    BodyToRenderGroupTable m_bodiesToRenderGroups;
    bool                   m_onlyAddEmittedBodies;
    // A placeholder for an empty visualization group
    RigidBodyRenderCache::RigidBodyVisualGroup m_emptyGroup;
  };
}
