/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/config/AGX_USE_AGXSENSOR.h>

#if AGX_USE_AGXSENSOR()

#include <agxOSG/export.h>

#include <agx/Vec4.h>
#include <agx/Quat.h>

#include <osg/Geode>
#include <osg/Geometry>
#include <osg/MatrixTransform>
#include <osg/PrimitiveSet>

#include <functional>

namespace agxSensor
{
  class Lidar;
}

namespace agxOSG
{
  /**
  Lidar point cloud renderer. Hit positions, including intensity, and optionally
  the rays of the lidar. Add this node to the scene graph to visualize the point cloud.
  */
  class AGXOSG_EXPORT LidarOutputRenderer : public osg::MatrixTransform
  {
    public:
      using GeodeRef = osg::ref_ptr<osg::Geode>;
      using GeometryRef = osg::ref_ptr<osg::Geometry>;
      using Vec3ArrayRef = osg::ref_ptr<osg::Vec3Array>;
      using Vec4ArrayRef = osg::ref_ptr<osg::Vec4Array>;
      using ColorArrayRef = osg::ref_ptr<osg::Vec4Array>;
      using ProgramRef = osg::ref_ptr<osg::Program>;
      using UniformRef = osg::ref_ptr<osg::Uniform>;

      struct IntensityState
      {
        using ColorFunc = std::function<osg::Vec4( float intensity )>;

        enum Mode
        {
          COLOR_RG, /**< Color red (low intensity) -> green (high intensity). (Default) */
          COLOR_A, /**< Color alpha - hit points with intensity as transparency. */
        };

        float scale = 120.0f;
        ColorFunc colorFunc = {};
      };

      class LidarObserverHolder;

    public:
      /**
      Construct given lidar and option to render the lidar rays (default: false).
      \param lidar - lidar instance
      \param renderRays - true to render rays of the lidar, false to only visualize the point cloud
      */
      LidarOutputRenderer( const agxSensor::Lidar* lidar, bool renderRays = false );

      /**
      Set scale of the circle representing a hit. Default: 1.0.
      */
      void setHitPointScale( float scale );

      /**
      \return the current scale of the circle representing a hit (defaualt: 1.0)
      */
      float getHitPointScale() const;

      /**
      Set mode how the intenstity of a hit is visualized - red (low) to green (high), or alpha.
      \param mode - new intensity visualization mode
      */
      void setIntensityMode( IntensityState::Mode mode );

      /**
      Set scale of the output intensity of the lidar. This scale transforms the lidar
      output intensity to color space [0, 1]. Default: 120.0.
      \param intensityScale - new intensity scale, default: 120.0
      */
      void setIntensityScale( float intensityScale );

      /**
      Set custom color function of the hits. The input is the scaled (setIntensityScale) intensity
      of the hit point, the output is the color RGBA (osg::Vec4).
      \param colorFunc - new hit point color function
      */
      void setIntensityColorFunction( IntensityState::ColorFunc colorFunc );

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      struct HitOutput : agx::Vec4f
      {
        using agx::Vec4f::Vec4f;

        agx::Vec3f position() const
        {
          return { x(), y(), z() };
        }

        osg::Vec3 osgPosition() const
        {
          return { x(), y(), z() };
        }

        osg::Vec4 osgZToNormal( agx::Quat32 toWorld ) const
        {
          agx::Quat32 q = agx::Quat32{ agx::Vec3f::Z_AXIS(), normal } * toWorld.conj();
          return { q.x(), q.y(), q.z(), q.w() };
        }

        float intensity() const
        {
          return w();
        }

        agx::Vec3f normal;
      };

      struct FrameCallback : osg::NodeCallback
      {
        virtual void operator()(osg::Node* node, osg::NodeVisitor* /*visitor*/) override
        {
          dynamic_cast<LidarOutputRenderer*>(node)->onPreRender();
        }
      };

      struct CloudRenderData
      {
        CloudRenderData(size_t maxNumPoints);
        GeometryRef createGeometry(osg::PrimitiveSet::Mode renderMode);
        osg::StateSet* getCommonStateSet();

        void invalidate();

        GeodeRef geode;
        Vec3ArrayRef vertices;
        ColorArrayRef colors;
        Vec4ArrayRef zToNormals;
      };

      struct GeometryShaderData
      {
        GeometryRef rayGeometry;
        ProgramRef rayProgram;
        UniformRef rayColor;

        GeometryRef pointsGeometry;
        ProgramRef pointsProgram;
        UniformRef hitPointScale;
      };

    public:
        LidarOutputRenderer( const LidarOutputRenderer& object );
        virtual ~LidarOutputRenderer() noexcept;

        LidarOutputRenderer& operator=( const LidarOutputRenderer& object );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      friend FrameCallback;

      void synchronizeTransform();
      ProgramRef createShader(const agx::String& filename);

      void initializeOutputCloudGeometryShader( bool renderRays );
      void onPreRender();

    private:
      LidarObserverHolder* m_lidarObserver; // This is fully managed by the LidarOutputRenderer
      CloudRenderData m_cloudRenderData;
      GeometryShaderData m_geometryShaderData;
      IntensityState m_intensityState;
  };

  using LidarOutputRendererRef = osg::ref_ptr<LidarOutputRenderer>;
}

#endif // AGX_USE_AGXSENSOR()
