/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/Constraint.h>
#include <agx/LSquareComplianceMatrix.h>
#include <agx/agx_vector_types.h>

namespace agx
{
  /**
  Compliance matrix data containing the lower triangle (including the diagonal)
  of the compliance matrix of the constraint. This matrix can be written to by
  calling self.setSupportComplianceMatrix( True ) and implement:

  def updateComplianceMatrix(self, matrix: agx.GenericConstraintComplianceMatrix):
      matrix.set(1, 0, 1.0E-4)
  */
  class AGXPHYSICS_EXPORT GenericConstraintComplianceMatrix : public agx::Referenced
  {
    public:
      /**
      Construct given maximum size, i.e., when all elementary constraint
      rows are active and enabled, and if the size is constant, i.e., all
      elementary constraints are always active and enabled.
      */
      GenericConstraintComplianceMatrix( UInt32 maxSize, Bool isStaticSize );

      /**
      \param row - matrix row number
      \param col - matrix column number
      \return value at given row and column
      */
      Real get( UInt32 row, UInt32 col ) const;

      /**
      Assign value to given row and column in the matrix.
      \param row - matrix row number
      \param col - matrix column number
      */
      void set( UInt32 row, UInt32 col, Real value );

      /**
      \return the size, i.e., N in this N x N matrix
      */
      UInt32 size() const;

      /**
      Reset this matrix to zero and resize. If this matrix is of static
      size and \p size != current size, an exception is thrown.
      \param size - new size of this matrix
      */
      void reset( UInt32 size );

      /**
      \return vector of diagonal values that has been assigned
      */
      const UInt32Vector& getDiagonalUpdates() const;

      /**
      \return true if this matrix has a static size, otherwise false
      */
      Bool isStaticSize() const;

    protected:
      /**
      Reference counted object, protected destructor.
      */
      virtual ~GenericConstraintComplianceMatrix();

    private:
      /**
      Calculates index in m_data given matrix row and column.
      */
      UInt32 calculateIndex( UInt32 row, UInt32 col ) const;

      /**
      \return array storage size
      */
      UInt32 arraySize() const;

    private:
      RealVector m_data;
      UInt32 m_size;
      UInt32 m_maxSize;
      Bool m_isStaticSize;
      UInt32Vector m_diagonalUpdates;
  };

  using GenericConstraintComplianceMatrixRef = agx::ref_ptr<GenericConstraintComplianceMatrix>;
}
