
#pragma once

#include <memory>
#include <string>
#include <vector>
#include <openplx/Token.h>
#include <openplx/Node.h>

namespace openplx
{
    class VarAssignment : public Node, public std::enable_shared_from_this<VarAssignment>
    {
        public:
            /* Factories */
            static VarAssignPtr create(
                const std::vector<Token>& target_segments,
                const Token& type_modifier,
                const std::vector<Token>& type_segments,
                const Token& post_modifier,
                const std::vector<AnnotationPtr>& annotations,
                const std::vector<VarAssignPtr>& members,
                ExprPtr value);

            /* Overrides */
            VarAssignPtr asVarAssignment() override;
            void accept(NodeVisitor& visitor) override;
            void unbind() override;

            /* Target */
            std::string targetSegmentsAsString() const;
            std::string completeTargetSegmentsAsString() const;
            std::string targetSegmentsAsStringSkipLast() const;
            std::string targetSegmentsUpToAsString(size_t index) const;
            std::string completeTargetSegmentsAsStringSkipLast() const;
            const std::vector<Token>& getTargetSegments() const;
            std::vector<Token> getCompleteTargetSegments() const;

            /* Type modifier (is or becomes) */
            Token getTypeModifier() const;

            /* Type */
            std::string typeSegmentsAsString() const;
            const std::vector<Token>& getTypeSegments() const;
            const std::vector<VarAssignPtr>& getTargetTypes() const;
            std::vector<VarAssignPtr> getCompleteTargetTypes() const;
            void setTargetTypes(const std::vector<VarAssignPtr>& type);
            TypePtr getType() const;
            TypePtr getResolvedType() const;
            void setType(TypePtr type);

            /* Suffix modifier (keyword after type before ':') */
            Token getPostModifier() const;

            /* Annotations */
            const std::vector<AnnotationPtr>& getAnnotations() const;
            void appendToAnnotations(AnnotationPtr annotation);

            /* Members */
            const std::vector<VarAssignPtr>& getMembers() const;
            void appendToMembers(VarAssignPtr member);

            /* Value */
            ExprPtr getValue() const;
            void setValue(ExprPtr value);

            /* Owning document */
            DocPtr getOwningDocument() const;
            void setOwningDocument(DocPtr owning_document);

            /* Owning model */
            ModelDeclPtr getOwningModel() const;
            void setOwningModel(ModelDeclPtr owning_model);

            /* Owning declaration */
            VarAssignPtr getOwningDeclaration() const;
            void setOwningDeclaration(VarAssignPtr owning_declaration);

            /* Source id */
            std::string getSourceIdOrDefault() const;

            /* Utility */
            bool isRootVarDeclaration() const;
            bool isNestedDeclaration() const;
            bool isReference() const;
            bool isReferenceOrDescendant() const;
            bool hasNestedDeclarations() const;
            bool hasType() const;
            bool hasWithType() const;
            bool isExtended() const;
            std::string getNameWithNamespace(const std::string& separator) const;
            VarAssignPtr followFirstType();
            VarAssignPtr clone() const;

            // @deprecated kept until all trace of former VarDeclaration is removed
            bool isFormerVarDeclaration() const;

        private:
            VarAssignment();

            void collectCompleteTargetTypes(std::vector<VarAssignPtr>& target_types) const;
            void collectCompleteTargetSegments(std::vector<Token>& target_segments) const;

            std::vector<Token> m_target_segments;
            Token m_type_modifier;
            std::vector<Token> m_type_segments;
            Token m_post_modifier;
            std::vector<VarAssignPtr> m_target_types;
            TypePtr m_type;
            std::vector<AnnotationPtr> m_annotations;
            std::vector<VarAssignPtr> m_members;
            ExprPtr m_value;
            DocPtr m_owning_document;
            ModelDeclPtr m_owning_model;
            VarAssignPtr m_owning_declaration;
    };
}
