#pragma once

#include <optional>
#include <string>
#include <memory>
#include <unordered_map>
#include <deque>
#include <sstream>
#include <openplx/Strings.h>
#include <openplx/Token.h>

namespace openplx
{
    class ModelDeclaration;
    template <typename ValueType>
    class TreeNode : public std::enable_shared_from_this<TreeNode<ValueType>> {
        public:
            static std::shared_ptr<TreeNode<ValueType>> create(
                std::shared_ptr<ValueType> value = nullptr)
            {
                auto node = new TreeNode();
                node->m_value = std::move(value);
                return std::shared_ptr<TreeNode<ValueType>>(node);
            }

            const std::shared_ptr<ValueType>& getValue() const { return m_value; }
            void setValue(std::shared_ptr<ValueType> value) { m_value = std::move(value); }
            void collectValues(std::unordered_set<std::shared_ptr<ValueType>>& results) {
                if (m_value != nullptr) {
                    results.insert(m_value);
                }
                for (auto& child_entry : m_children) {
                    child_entry.second->collectValues(results);
                }
            }

            std::shared_ptr<TreeNode<ValueType>> clone() const {
                auto self_clone = TreeNode::create(m_value);
                for (auto& child : m_children) {
                    self_clone->m_children.emplace( child.first, child.second->clone() );
                }
                return self_clone;
            }

            template <class T>
            std::shared_ptr<TreeNode<ValueType>>
            insert(const std::deque<T>& path, std::shared_ptr<ValueType> value, size_t ix = 0, size_t subix = 0) {
                if (ix >= path.size()) {
                    m_value = std::move(value);
                    return this->shared_from_this();
                }
                if (subix == path[ix]->getTargetSegments().size()) {
                    return insert(path, value, ix + 1, 0);
                }
                auto it = m_children.find(path[ix]->getTargetSegments()[subix].lexeme);
                if (it == m_children.end()) {
                    auto new_branch = TreeNode::create();
                    m_children.emplace(path[ix]->getTargetSegments()[subix].lexeme, new_branch);
                    return new_branch->insert(path, std::move(value), ix, subix + 1);
                }
                return it->second->insert(path, std::move(value), ix, subix + 1);
            }

            std::shared_ptr<TreeNode<ValueType>>
            insert(const std::string& dot_seperated_path, std::shared_ptr<ValueType> value) {
                if (dot_seperated_path.empty()) {
                    m_value = std::move(value);
                    return this->shared_from_this();
                }
                auto segments = Internal::split(dot_seperated_path, ".");
                return insert(segments, std::move(value));
            }

            std::shared_ptr<TreeNode<ValueType>>
            insert(const std::vector<std::string>& path, std::shared_ptr<ValueType> value, size_t idx = 0) {
                if (idx == path.size()) {
                    m_value = std::move(value);
                    return this->shared_from_this();
                } else if (idx < path.size()) {
                    auto& child = m_children[path[idx]];
                    if (!child) {
                        child = TreeNode::create();
                        m_children[path[idx]] = child;
                    }
                    return child->insert(path, std::move(value), idx+1);
                }
                return nullptr;
            }

            std::shared_ptr<TreeNode<ValueType>>
            insert(const std::vector<Token>& path, std::shared_ptr<ValueType> value, size_t idx = 0) {
                if (idx == path.size()) {
                    m_value = std::move(value);
                    return this->shared_from_this();
                } else if (idx < path.size()) {
                    auto& child = m_children[path[idx].lexeme];
                    if (!child) {
                        child = TreeNode::create();
                        m_children.emplace(path[idx].lexeme, child);
                    }
                    return child->insert(path, std::move(value), idx+1);
                }
                return nullptr;
            }

            std::shared_ptr<TreeNode<ValueType>> lookupOrCreate(const std::string& key) {
                auto it = m_children.find(key);
                if (it == m_children.end()) {
                    auto branch = TreeNode::create();
                    m_children.emplace(key, branch);
                    return branch;
                }
                return it->second;
            }

            std::shared_ptr<TreeNode<ValueType>> lookup(const std::string& dot_seperated_path) {
                if (dot_seperated_path.empty()) {
                    return this->shared_from_this();
                }
                auto segments = Internal::split(dot_seperated_path, ".");
                return lookup(segments);
            }

            std::shared_ptr<TreeNode<ValueType>> lookup(const std::vector<std::string>& path) {
                return lookup(path, 0, path.size() - 1);
            }

            std::shared_ptr<TreeNode<ValueType>> lookup(const std::vector<Token>& path) {
                return lookup(path, 0, path.size() - 1);
            }

            std::shared_ptr<TreeNode<ValueType>> lookup(const std::vector<Token>& path, size_t first_ix) {
                return lookup(path, first_ix, path.size() - 1);
            }

            std::shared_ptr<TreeNode<ValueType>> lookup(const std::vector<std::string>& path, size_t first_ix) {
                return lookup(path, first_ix, path.size() - 1);
            }

            std::shared_ptr<TreeNode<ValueType>> lookup(const std::vector<std::string>& path, size_t first_ix, size_t last_ix) {
                if (first_ix == last_ix + 1) return this->shared_from_this();
                if (first_ix >= path.size()) return nullptr;
                auto it = m_children.find(path[first_ix]);
                if (it == m_children.end()) return nullptr;
                return it->second->lookup(path, first_ix+1, last_ix);
            }

            std::shared_ptr<TreeNode<ValueType>> lookup(const std::vector<Token>& path, size_t first_ix, size_t last_ix) {
                if (first_ix == last_ix + 1) return this->shared_from_this();
                if (first_ix >= path.size()) return nullptr;
                auto it = m_children.find(path[first_ix].lexeme);
                if (it == m_children.end()) return nullptr;
                return it->second->lookup(path, first_ix+1, last_ix);
            }

            template <class T>
            std::shared_ptr<TreeNode<ValueType>> lookup(const std::deque<T>& path, size_t ix = 0, size_t subix = 0) {
                if (ix >= path.size()) {
                    return this->shared_from_this();
                }
                if (subix == path[ix]->getTargetSegments().size()) {
                    return lookup(path, ix + 1, 0);
                }
                auto it = m_children.find(path[ix]->getTargetSegments()[subix].lexeme);
                if (it == m_children.end()) return nullptr;
                return it->second->lookup(path, ix, subix + 1);
            }

            std::shared_ptr<ValueType> lookupValue(const std::string& dot_seperated_path) {
                if (dot_seperated_path.empty()) {
                    return m_value;
                }
                auto segments = Internal::split(dot_seperated_path, ".");
                return lookupValue(segments);
            }

            std::shared_ptr<ValueType> lookupValue(const std::vector<std::string>& path) {
                return lookupValue(path, 0, path.size() - 1);
            }

            std::shared_ptr<ValueType> lookupValue(const std::vector<Token>& path) {
                return lookupValue(path, 0, path.size() - 1);
            }

            std::shared_ptr<ValueType> lookupValue(const std::vector<Token>& path, size_t first_ix) {
                return lookupValue(path, first_ix, path.size() - 1);
            }

            std::shared_ptr<ValueType> lookupValue(const std::vector<std::string>& path, size_t first_ix) {
                return lookupValue(path, first_ix, path.size() - 1);
            }

            std::shared_ptr<ValueType> lookupValue(const std::vector<std::string>& path, size_t first_ix, size_t last_ix) {
                auto node = lookup(path, first_ix, last_ix);
                if (node == nullptr) return nullptr;
                return node->m_value;
            }

            std::shared_ptr<ValueType> lookupValue(const std::vector<Token>& path, size_t first_ix, size_t last_ix) {
                auto node = lookup(path, first_ix, last_ix);
                if (node == nullptr) return nullptr;
                return node->m_value;
            }

            template <class T>
            std::shared_ptr<ValueType> lookupValue(const std::deque<T>& path, size_t ix = 0, size_t subix = 0) {
                auto node = lookup(path, ix, subix);
                if (node == nullptr) return nullptr;
                return node->m_value;
            }

            const std::unordered_map<std::string, std::shared_ptr<TreeNode<ValueType>>>& getChildren() {
                return m_children;
            }

            std::string toString(std::string indent) {
                std::ostringstream oss;
                if (m_value != nullptr) {
                    oss << m_value->toString(indent) << std::endl;
                }
                for (auto& child_entry : m_children) {
                    oss << indent << "Key: " << child_entry.first << std::endl;
                    oss << child_entry.second->toString(indent + "  ");
                }
                return oss.str();
            }

            void cloneInto(std::shared_ptr<TreeNode<ValueType>> root) const {
                root->m_value = m_value->clone();
                for (auto& child_entry : m_children) {
                    auto it = root->m_children.find(child_entry.first);
                    if (it == root->m_children.end()) {
                        auto branch = TreeNode<ValueType>::create();
                        root->m_children.emplace(child_entry.first, branch);
                        child_entry.second->cloneInto(branch);
                    } else {
                        child_entry.second->cloneInto(it->second);
                    }
                }
            }

            void unbind() {
                for (auto& pair : m_children) {
                    pair.second->unbind();
                }
                m_children.clear();
                m_value.reset();
            }
        private:
            TreeNode() = default;
            std::shared_ptr<ValueType> m_value;
            std::unordered_map<std::string, std::shared_ptr<TreeNode<ValueType>>> m_children;
    };
}
