#pragma once

#include <string>
#include <openplx/TokenType.h>
#include <openplx/Types.h>

namespace openplx
{
    struct Token {
        TokenType type;
        std::string lexeme;
        line_t line;
        col_t column;

        Token() : type(TokenType::Uninitialized), lexeme(), line(0), column(0) {}

#ifdef SWIGCSHARP
        Token(TokenType _type, std::string _lexeme, line_t _line = 0, col_t _column = 0)
          : type(_type), lexeme(std::move(_lexeme)), line(_line), column(_column) {}

#else
        Token(TokenType _type, std::string&& _lexeme, line_t _line = 0, col_t _column = 0)
          : type(_type), lexeme(std::move(_lexeme)), line(_line), column(_column) {}
#endif

        static Token Uninitialized() {
            return {};
        }
        static Token Invalid() {
            return { TokenType::Invalid, "" };
        }

        static Token Identifier(std::string lexeme) {
            return { TokenType::Identifier, std::move(lexeme) };
        }

        static Token Empty() {
            return { TokenType::NoneType, "" };
        }

        static Token Keyword(TokenType type) {
            switch (type) {
                case TokenType::Is:
                    return { type, "is" };
                case TokenType::As:
                    return { type, "as" };
                case TokenType::Becomes:
                    return { type, "becomes" };
                case TokenType::Static:
                    return { type, "static" };
                case TokenType::Const:
                    return { type, "const" };
                case TokenType::Fn:
                    return { type, "fn" };
                case TokenType::Operator:
                    return { type, "operator" };
                case TokenType::Trait:
                    return { type, "trait" };
                case TokenType::With:
                    return { type, "with" };
                case TokenType::Delete:
                    return { type, "delete" };
                case TokenType::Reference:
                    return { type, "reference" };
                default:
                    return { type, "" };
            }
        }

        static Token Bool(bool value) {
            if (value) {
                return { TokenType::TrueKeyword, "true" };
            } else {
                return { TokenType::FalseKeyword, "false" };
            }
        }

        static Token Real(double value) {
            return { TokenType::Number, std::to_string(value) };
        }

        static Token Int(int64_t value) {
            return { TokenType::Number, std::to_string(value) };
        }

        static Token String(std::string value) {
            return { TokenType::String, std::move(value) };
        }

        static std::pair<line_t, col_t> end_of(const Token& token) {
            if (token.type == TokenType::MultiLineString) {
                line_t line = token.line;
                col_t column = token.column;
                for (auto& c : token.lexeme) {
                    if (c == '\n') {
                        line++;
                        column = 1;
                    } else {
                        column++;
                    }
                }
                return { line, column };
            } else {
                return { token.line, token.column + token.lexeme.size() };
            }
        }

        static std::string extractString(const Token& token) {
            if (token.type == TokenType::String) {
                return token.lexeme.substr(1, token.lexeme.size() - 2);
            }
            if (token.type == TokenType::MultiLineString) {
                return token.lexeme.substr(3, token.lexeme.size() - 6);
            }
            if (token.type == TokenType::AtString) {
                return token.lexeme.substr(2, token.lexeme.size() - 3);
            }
            return "";
        }

        bool operator==(const Token& other) const {
            return type == other.type &&
                   line == other.line &&
                   column == other.column &&
                   lexeme == other.lexeme;
        }
    };
}
