#pragma once

#include <memory>
#include <vector>
#include <spdlog/spdlog.h>
#include <spdlog/logger.h>

/**
 * Because SPDLOG is a header-only library, each Windows DLL will have its own instance of the logger.
 * This class is a workaround for setting up a default logger for the each DLL with custom sinks.
 * https://github.com/gabime/spdlog/wiki/How-to-use-spdlog-in-DLLs
 *
 * Note that if/when implementing this solution for all DLL:s, we need to
 * - Create a unique Logger class for each DLL
 */
namespace openplx {

class SPDLogWorkaround {
    protected:
        static inline const std::string logger_name = "logger";
        /**
         * setup_default_logger_impl must be inline:d so that each DLL can have it's own implementation of it.
         */
        static inline std::shared_ptr<spdlog::logger> setup_default_logger_impl(std::vector<spdlog::sink_ptr> sinks) {
            auto logger = spdlog::get(logger_name);
            if(!logger)
            {
                if(sinks.size() > 0)
                {
                    logger = std::make_shared<spdlog::logger>(logger_name,
                                                            std::begin(sinks),
                                                            std::end(sinks));
                    spdlog::set_default_logger(logger);
                }
            }

            return logger;
        }

        static inline std::shared_ptr<spdlog::logger> get_default_logger() {
            return spdlog::default_logger();
        }

        static inline void set_default_logger(std::shared_ptr<spdlog::logger> logger) {
            spdlog::set_default_logger(logger);
        }
};

}
