/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agx/config/AGX_USE_AGXTERRAIN.h>
#include <agxTerrain/export.h>
#include <agx/Referenced.h>
#include <agx/RigidBody.h>
#include <agxCollide/Geometry.h>
#include <agxTerrain/FailureSurface.h>


namespace agxTerrain
{
  // Forward declaration
  class Terrain;

  namespace GroundCollapse
  {

    AGX_DECLARE_POINTER_TYPES(FailureSurfaceAggregate);
    AGX_DECLARE_VECTOR_TYPES(FailureSurfaceAggregate);
    /*
    An aggregate representation of the soil inside the failure surface.
    */
    class AGXTERRAIN_EXPORT FailureSurfaceAggregate : public agx::Referenced
    {
    public:
      FailureSurfaceAggregate(const agx::String& name,
                    const agx::Vec3& centerOfMass,
                    agx::Real mass,
                    agx::Real radius,
                    const agx::SPDMatrix3x3& worldInertiaTensor,
                    const agx::Vec3iVector& voxels,
                    const agx::Vec3iVector& borderVoxels,
                    agx::Real voxelContactArea);

      inline agx::RigidBody* getBody() { return m_body; };
      inline agxCollide::Geometry* getGeometry() { return m_geometry; };
      inline agx::Real getMass() { return m_body->getMassProperties()->getMass(); };
      inline agx::SPDMatrix3x3 getInertiaTensor() { return m_body->getMassProperties()->getInertiaTensor(); };

      /*
      \return the contact area between the soil aggregate an the rest of the terrain.
      */
      agx::Real getVoxelContactArea() const;

      /*
      \return the radius of the soil aggregate
      */
      agx::Real getRadius() const;

      /*
      \return the solid voxels inside the soil aggregate.
      */
      agx::Vec3iVector getVoxels() const;

      /*
      \return the numer of solid voxels in the soil aggregate.
      */
      agx::UInt getNumVoxels() const;

      /*
      \return the subset of the voxels in the soil aggregate that borders the rest of the terrain.
      */
      agx::Vec3iVector getBorderVoxels() const;

    private:
      agx::RigidBodyRef         m_body;
      agxCollide::GeometryRef   m_geometry;
      agx::Vec3iVector          m_voxels;
      agx::Vec3iVector          m_borderVoxels;
      agx::Vec3                 m_initialPosition;
      agx::Vec3Vector           m_positionHistory;
      agx::Real                 m_contactArea;
    };

    AGXTERRAIN_EXPORT FailureSurfaceAggregate* constructSoilAggregate(
      const agx::Name& name,
      GroundCollapse::FailureSurface* failureSurface,
      const agxTerrain::Terrain* terrain );
  }
}
