/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxTerrain/PrimaryActiveZone.h>
#include <agx/Plane.h>
#include <agxTerrain/SoilParticleAggregate.h>
#include <agxTerrain/AggregateContactUtils.h>
#include <agxTerrain/TerrainFrictionModels.h>
#include <agxTerrain/ShovelContactPatchUtils.h>



namespace agxTerrain
{
  class Shovel;
  class ActiveZone;
  class Terrain;

  typedef agx::HashVector<agxCollide::Geometry*, agxCollide::LocalGeometryContact> GeometryLocalContactTable;
  typedef agx::HashSet<agxCollide::Geometry*> GeometryHashSet;

  AGX_DECLARE_POINTER_TYPES( AggregateContactGenerator );

  class AGXTERRAIN_EXPORT AggregateContactGenerator : public agx::Referenced
  {
  public:
    AggregateContactGenerator();

    void onPre(   Shovel* shovel,
                  Terrain* terrain    );

    void onPost( Shovel* shovel );

    /**
    \return the geometry contacts created between the shovel geometries and this aggregate
    */
    agxCollide::GeometryContactPtrVector getGeometryContacts( agxSDK::Simulation* simulation ) const;

    /**
    Initialize contact materials used in the aggregate contacts
    */
    void initializeContactMaterials( Terrain* terrain, Shovel* shovel );

    /**
    Initialize custom friction models used in the aggregate contacts
    */
    void initializeFrictionModels( Shovel* shovel );

    /**
    Resets contact model history in the contact generator
    */
    void resetAggregateContactHistory();

    /**
    Get the aggregate <-> terrain contact depth model
    */
    const AggregateContactDepthModel& getAggregateDepthModel() const;

    /**
    Checks if a particle with a given id is in contact with the shovel.
    \param particleId - the specified id of the particle to check.
    \return true if a particle with the specified id is in contact with the shovel, false otherwise.
    */
    bool particleIsInShovelContacts( agx::UInt32 particleId ) const;

    /**
    Get the internal shovel <-> aggregate contact material
    */
    agx::ContactMaterial* getAggregateShovelContactMaterial() const;

    /**
    Get the internal aggregate <-> terrain contact material
    */
    agx::ContactMaterial* getAggregateTerrainContactMaterial() const;

    /**
    \return the maximum number of contact points given the current
    partition data and the possibility for side contacts.
    */
    size_t getMaximumNumberOfContactPoints() const;

    /**
    \return the number of current contact points between the shovel and the
    inner shape aggregate.
    */
    size_t getNumShovelContactPoints() const;

    DOXYGEN_START_INTERNAL_BLOCK()

    /**
    * Create contact partition data for the specified shovel.
    */
    void createContactPartitions( Shovel* shovel );

    void setEnableCustomFrictionModels( bool enable );
    bool getEnableCustomFrictionModels() const;

    DOXYGEN_END_INTERNAL_BLOCK()

  protected:
    virtual ~AggregateContactGenerator();

  private:

    void createGeometryContacts(SoilParticleAggregate* soilParticleAggregate,
                                PrimaryActiveZone* activeZone,
                                Shovel* shovel,
                                Terrain* terrain);

    void createTerrainContacts( SoilParticleAggregate* soilParticleAggregate,
                                PrimaryActiveZone* activeZone,
                                Shovel* shovel,
                                Terrain* terrain,
                                agxCollide::LocalGeometryContactVector& localContacts );

    void createShovelContacts(SoilParticleAggregate* soilParticleAggregate,
                              Shovel* shovel,
                              Terrain* terrain,
                              agxCollide::LocalGeometryContactVector& localContacts);

    std::tuple<agx::Real, agx::UInt> getContactPointAngle(
      const agx::Vec3& midPoint,
      const agx::Vec3& zeroPoint,
      const agx::Vec3& contactPoint,
      const agx::Plane& polarPlane,
      const agx::Plane& cutTopPlane ) const;

    GeometryHashSet getShovelGeometries( Shovel* shovel ) const;

    void getShovelContactsAndContactRange( Shovel* shovel,
                                           const agx::Vector<agx::Plane>& shovelPlanes,
                                           GeometryHashSet& geometrySet,
                                           agxCollide::ParticleGeometryContactVector& contacts,
                                           agxCollide::ParticleGeometryContactVector& rightWallContacts,
                                           agxCollide::ParticleGeometryContactVector& leftWallContacts,
                                           agx::RangeReal& range );

    agxCollide::LocalParticleGeometryContactVector calculateShovelContactPoints( Shovel* shovel,
                                                                                 const agxCollide::ParticleGeometryContactVector& shovelContacts,
                                                                                 const agx::RangeReal& range,
                                                                                 agx::Real& minContactArea ) const;

    void calculateShovelSideWallContactPoints(
      Shovel* shovel,
      const agx::Vector<agx::Plane>& shovelPlanes,
      GeometryHashSet& shovelGeometryTable,
      GeometryLocalContactTable& shovelContactTable,
      agxCollide::ParticleGeometryContactVector& rightWallContacts,
      agxCollide::ParticleGeometryContactVector& leftWallContacts );

    void updateContactSpan( agxTerrain::Shovel* shovel,
                            ContactSpanData& contactSpanData,
                            const agx::Physics::ParticleGeometryContactPtr& contact,
                            const agx::Plane& middlePlane,
                            agx::Real contactAngle ) const;

    agxCollide::LocalParticleGeometryContactVector constructShovelContacts( Shovel* shovel,
                                                                            agx::Vector<ContactPatch>& contactPatchData,
                                                                            ContactSpanData& contactSpanData,
                                                                            const agx::Vec3f& separationNormal,
                                                                            const agx::Vec3& cuttingVector ) const;

    agx::Real calculateContactArea( agx::Real minContactArea,
                                    const Shovel* shovel,
                                    const agxCollide::LocalParticleGeometryContactVector& contacts ) const;

    agx::UInt getNumContactPatches() const { return m_partitionData.getNumPatches(); };

    void debugRenderContactPatches( agxTerrain::Shovel* shovel ) const;

    void updateContactPartitions( agxTerrain::Shovel* shovel );

  private:
    using ParticleIdSet = agx::HashSet<agx::UInt32>;

    ParticleIdSet                          m_particleIds;
    ParticleIdSet                          m_particlesInShovelContacts;
    agxCollide::LocalGeometryContactVector m_localGeometryContacts;
    agxCollide::LocalGeometryContactVector m_localShovelAggregateContacts;
    agx::ContactMaterialRef                m_aggregateTerrainContactMaterial;
    agx::ContactMaterialRef                m_aggregateShovelContactMaterial;
    AggregateContactDepthModel             m_depthModel;
    ShovelAggregateFrictionModelRef        m_frictionModel;
    bool                                   m_enableCustomFrictionModels;
    ContactPartitionData                   m_partitionData;
    ContactPatchVector                     m_patchData;

  };
}
