/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/export.h>

#include <agx/Range.h>
#include <agx/Vec3.h>

namespace agxSensor
{
  /**
  Range specification along three axis.
  */
  class AGXSENSOR_EXPORT TriaxialRange
  {
    public:
      /**
      Constructs a triaxial range by extending the given \p range to each axis.
      \param range - range for all of the axis
      */
      TriaxialRange( agx::RangeReal range );

      /**
      Constructs a triaxial range from the given ranges for each of the axis.
      \param rangeX - range along the x-axis
      \param rangeY - range along the y-axis
      \param rangeZ - range along the z-axis
      */
      TriaxialRange( agx::RangeReal rangeX,
                     agx::RangeReal rangeY,
                     agx::RangeReal rangeZ );

      /**
      Constructs a triaxial range with the given bounds.
      \param lower - lower bound (may be negative)
      \param upper - upper bound
      */
      TriaxialRange( agx::Vec3 lower = agx::Vec3( std::numeric_limits<agx::Real>::lowest() ),
                     agx::Vec3 upper = agx::Vec3( std::numeric_limits<agx::Real>::max() ) );

      /**
      \return the lower bound of the range
      */
      agx::Vec3 getLower() const;

      /**
      \return the upper bound of the range
      */
      agx::Vec3 getUpper() const;

      /**
      \return the range, lower to upper, along the x-axis
      */
      agx::RangeReal getRangeX() const;

      /**
      \return the range, lower to upper, along the y-axis
      */
      agx::RangeReal getRangeY() const;

      /**
      \return the range, lower to upper, along the z-axis
      */
      agx::RangeReal getRangeZ() const;

    private:
      agx::Vec3 m_rangeLower, m_rangeUpper;
  };
}
