/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/SensorOutput.h>
#include <agxSensor/SystemNode.h>

#include <agx/Referenced.h>

#include <limits>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( IMUOutput );

  /**
  IMU sensor output data where IMUOutput::Field is used to define the data available.

  It's possible to pack data into a custom type, e.g.,

    struct MyIMUOutput
    {
      agx::Real accelerometer;
      agx::Real gyroscopeRoll, gyroscopePitch;
    }

    IMUOutputRef output = new IMUOutput();
    output->build<IMUOutput::ACCELEROMETER_Z_F64,
                  IMUOutput::GYROSCOPE_X_F64,
                  IMUOutput::GYROSCOPE_Y_F64>();

    ... // Run sensor

    auto view = output->view<MyIMUOutput>();
    for ( const MyIMUOutput& value : view )
      ...
  */
  class AGXSENSOR_EXPORT IMUOutput : public SystemNode, public ISensorOutput
  {
    public:
      enum Field : int32_t
      {
        /** 1x 64-bit floating point accelerometer x-axis output. */
        ACCELEROMETER_X_F64,
        /** 1x 64-bit floating point accelerometer y-axis output. */
        ACCELEROMETER_Y_F64,
        /** 1x 64-bit floating point accelerometer z-axis output. */
        ACCELEROMETER_Z_F64,
        /** 1x 64-bit floating point gyroscope x-axis output. */
        GYROSCOPE_X_F64,
        /** 1x 64-bit floating point gyroscope y-axis output. */
        GYROSCOPE_Y_F64,
        /** 1x 64-bit floating point gyroscope z-axis output. */
        GYROSCOPE_Z_F64,
        /** 1x 64-bit floating point magnetometer x-axis output. */
        MAGNETOMETER_X_F64,
        /** 1x 64-bit floating point magnetometer y-axis output. */
        MAGNETOMETER_Y_F64,
        /** 1x 64-bit floating point magnetometer z-axis output. */
        MAGNETOMETER_Z_F64,
        /** 1x 64-bit floating point x-axis output from the first sensor in the IMU. */
        SENSOR_0_X_F64,
        /** 1x 64-bit floating point y-axis output from the first sensor in the IMU. */
        SENSOR_0_Y_F64,
        /** 1x 64-bit floating point z-axis output from the first sensor in the IMU. */
        SENSOR_0_Z_F64,
        /** 1x 64-bit floating point x-axis output from the second sensor in the IMU. */
        SENSOR_1_X_F64,
        /** 1x 64-bit floating point y-axis output from the second sensor in the IMU. */
        SENSOR_1_Y_F64,
        /** 1x 64-bit floating point z-axis output from the second sensor in the IMU. */
        SENSOR_1_Z_F64,
        SENSOR_2_X_F64,
        SENSOR_2_Y_F64,
        SENSOR_2_Z_F64,
        SENSOR_3_X_F64,
        SENSOR_3_Y_F64,
        SENSOR_3_Z_F64,
        SENSOR_4_X_F64,
        SENSOR_4_Y_F64,
        SENSOR_4_Z_F64,
        SENSOR_5_X_F64,
        SENSOR_5_Y_F64,
        SENSOR_5_Z_F64,
        SENSOR_6_X_F64,
        SENSOR_6_Y_F64,
        SENSOR_6_Z_F64,
        SENSOR_7_X_F64,
        SENSOR_7_Y_F64,
        SENSOR_7_Z_F64,
        SENSOR_8_X_F64,
        SENSOR_8_Y_F64,
        SENSOR_8_Z_F64,
        SENSOR_9_X_F64,
        SENSOR_9_Y_F64,
        SENSOR_9_Z_F64,
        /** 1x 64-bit floating point x-axis output from the last possibly added sensor. */
        SENSOR_MAX_X_F64 = ( ( ( 1 << 30 ) - 3 ) / 3) * 3,
        /** 1x 64-bit floating point y-axis output from the last possibly added sensor. */
        SENSOR_MAX_Y_F64 = SENSOR_MAX_X_F64 + 1,
        /** 1x 64-bit floating point z-axis output from the last possibly added sensor. */
        SENSOR_MAX_Z_F64 = SENSOR_MAX_X_F64 + 2,
        /** (Number of fields indicator) */
        NUM_FIELDS
      };

      using FieldVector = std::vector<Field>;

    public:
      /**
      \param field - data field
      \return data size of the field in bytes
      */
      static size_t sizeOf( Field field );

      /**
      \param field - data field
      \return name of the given field
      */
      static const char* getFieldName( Field field );

      /**
      Default constructor.
      */
      IMUOutput() = default;

      /**
      Add one or more fields, build a matching data structure.
      */
      template<Field... field>
      void build();

      /**
      Add new field.
      \param field - field to add
      \return reference to this IMU output
      */
      IMUOutput& add( Field field );

      /**
      Set value to use for padding when no underlying signal data can be found.
      \param value - value to use when padding in case of missing underlying signal
      */
      void setPaddingValue( const agx::Real& value );

      /**
      "Element size" is the summed data size of the currently added fields.
      \return the current, summed, data size of the added fields
      */
      virtual size_t getElementSize() const override;

      /**
      \return IMU output data as a pure binary buffer
      */
      virtual const BinaryOutputBuffer& getData() override;

      /**
      \param markAsRead - true to reset unread state, resulting in hasUnreadData returning
                          false until new data is available. False to not mark the current
                          data as read.
      \return true when the data has been updated but not read/used/fetched by
              the user, false if the data has been read
      */
      virtual bool hasUnreadData( bool markAsRead = true ) override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~IMUOutput() = default;

      virtual void result( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::IMUOutput );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      FieldVector m_fields{};
      size_t m_elementSize{ 0u };
      agx::Real m_paddingValue{ std::numeric_limits<agx::Real>::signaling_NaN() };
      bool m_hasUnreadData{ false };
      BinaryOutputBuffer m_output{};
  };

  template<IMUOutput::Field... field>
  void IMUOutput::build()
  {
    m_fields.clear();
    m_elementSize = 0u;
    ( this->add( field ), ... );
  }

  DOXYGEN_START_INTERNAL_BLOCK() // swig

  #pragma pack(push, 1)
  struct SixDoFValue
  {
    agx::Real x0, y0, z0;
    agx::Real x1, y1, z1;

    inline SixDoFValue() {}
    inline SixDoFValue( const agx::Vec3& a, const agx::Vec3& b )
      : x0( a.x() )
      , y0( a.y() )
      , z0( a.z() )
      , x1( b.x() )
      , y1( b.y() )
      , z1( b.z() )
    {
    }
    inline SixDoFValue( const SixDoFValue& ) = default;
    inline SixDoFValue& operator=( const SixDoFValue& ) = default;

    inline agx::Vec3 getTriplet( const size_t i ) const noexcept
    {
      if ( i == 0 )
        return agx::Vec3( x0, y0, z0 );
      else if ( i == 1 )
        return agx::Vec3( x1, y1, z1 );
      else
        return agx::Vec3();
    }
  };

  struct NineDoFValue
  {
    agx::Real x0, y0, z0;
    agx::Real x1, y1, z1;
    agx::Real x2, y2, z2;

    inline NineDoFValue() {}
    inline NineDoFValue( const agx::Vec3& a, const agx::Vec3& b, const agx::Vec3& c )
      : x0( a.x() )
      , y0( a.y() )
      , z0( a.z() )
      , x1( b.x() )
      , y1( b.y() )
      , z1( b.z() )
      , x2( c.x() )
      , y2( c.y() )
      , z2( c.z() )
    {
    }
    inline NineDoFValue( const NineDoFValue& ) = default;
    inline NineDoFValue& operator=( const NineDoFValue& ) = default;

    inline agx::Vec3 getTriplet( const size_t i ) const noexcept
    {
      if ( i == 0 )
        return agx::Vec3( x0, y0, z0 );
      else if ( i == 1 )
        return agx::Vec3( x1, y1, z1 );
      else if ( i == 2 )
        return agx::Vec3( x2, y2, z2 );
      else
        return agx::Vec3();
    }
  };
  #pragma pack(pop)

  class AGXSENSOR_EXPORT IMUOutputSixDoF : public IMUOutput
  {
    public:
      IMUOutputSixDoF();
  };

  class AGXSENSOR_EXPORT IMUOutputNineDoF : public IMUOutput
  {
    public:
      IMUOutputNineDoF();
  };

  DOXYGEN_END_INTERNAL_BLOCK()
}
