/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxSensor/RaytraceHandles.h>
#include <agxSensor/RaytraceRegistry.h>
#include <agxSensor/RaytraceInstanceData.h>

#include <agxVehicle/Track.h>

#include <agx/RigidBody.h>
#include <agx/InternalData.h>
#include <agx/BitState.h>

namespace agxSensor
{
  DOXYGEN_START_INTERNAL_BLOCK()

  class Environment;
  static constexpr auto SENSOR_DATA_SOURCE = agx::InternalData::SENSOR_ENVIRONMENT;

  struct AGXSENSOR_EXPORT RtInternalData : RtInstanceData
  {
    struct State
    {
      enum Flag : agx::UInt32
      {
        ENTITY_ID_OWNER = 1 << 0, /**< Id owners will call registry to destroy its data when this
                                       class is destructed. */
        MATERIAL_OWNER = 1 << 1,  /**< Material owners won't get new ones when overrideInherited = false. */
      };

      agx::BitState<Flag, agx::UInt32> flags;
    };

    using RtInstanceData::RtInstanceData;

    virtual ~RtInternalData()
    {
      if ( getEntityId() != NullRtEntityId && state.flags.Is( State::ENTITY_ID_OWNER ) ) {
        destroyEntityId();
        state.flags.update( State::ENTITY_ID_OWNER, false );
      }
    }

    State state;
  };
  using RtInternalDataRef = agx::ref_ptr<RtInternalData>;

  struct EnvInternalData : agx::Referenced
  {
    RtInternalDataRef rtData;

    Environment* environment = nullptr;
  };

  template<typename T>
  inline const EnvInternalData* getEnvData( const T& instance )
  {
    return agx::InternalData::get<EnvInternalData>( instance,
                                                    SENSOR_DATA_SOURCE );
  }

  template<typename T>
  inline EnvInternalData* getEnvData( T&& instance )
  {
    return const_cast<EnvInternalData*>( getEnvData( (const T&)instance ) );
  }

  template<typename T>
  inline EnvInternalData* getOrCreateEnvData( T&& instance )
  {
    return agx::InternalData::getOrCreate<EnvInternalData>( instance,
                                                            SENSOR_DATA_SOURCE );
  }

  template<typename T>
  inline const RtInternalData* getRtData( const T& instance )
  {
    const auto envData = getEnvData( instance );
    return envData != nullptr ?
             envData->rtData :
             nullptr;
  }

  template<typename T>
  inline RtInternalData* getOrCreateRtData( T&& instance )
  {
    if ( instance == nullptr )
      return nullptr;

    // Important that it's a reference to the pointer when
    // it's assigned in the return statement.
    auto& rtData = getOrCreateEnvData( instance )->rtData;
    return rtData != nullptr ?
             rtData.get() :
             ( rtData = new RtInternalData() ).get();
  }

  template<typename T>
  bool isRtMaterialOwner( const T& instance )
  {
    const RtInternalData* rtData = getRtData( instance );
    return rtData != nullptr &&
           rtData->state.flags.Is( RtInternalData::State::MATERIAL_OWNER );
  }

  template<typename T>
  void updateRtMaterial( T&& instance, RtMaterialInstance material )
  {
    if ( instance == nullptr )
      return;

    RtInternalData* rtData = getOrCreateRtData( instance );
    rtData->setMaterial( material );
  }

  AGXSENSOR_EXPORT void updateRtMaterial( agxVehicle::Track* track , RtMaterialInstance material );

  inline void updateRtMaterial( RtInternalData* rtData,
                                RtMaterialInstance material,
                                bool overrideInherited )
  {
    // Use if override or if not owner of a material.
    const auto useGivenHandle = overrideInherited ||
                                !rtData->state.flags.Is( RtInternalData::State::MATERIAL_OWNER );
    if ( useGivenHandle ) {
      rtData->setMaterial( material );
      rtData->state.flags.update( RtInternalData::State::MATERIAL_OWNER, overrideInherited );
    }
  }

  inline void updateRtMaterial( agxCollide::Shape* shape,
                                RtMaterialInstance material,
                                bool overrideInherited )
  {
    if ( shape == nullptr )
      return;

    updateRtMaterial( getOrCreateRtData( shape ),
                      material,
                      overrideInherited );
  }

  inline void updateRtMaterial( agxCollide::Geometry* geometry,
                                RtMaterialInstance material,
                                bool overrideInherited )
  {
    if ( geometry == nullptr )
      return;

    updateRtMaterial( getOrCreateRtData( geometry ),
                      material,
                      overrideInherited );

    for ( const auto& shape : geometry->getShapes() )
      updateRtMaterial( shape, material, false );
  }

  inline void updateRtMaterial( agx::RigidBody* rb,
                                RtMaterialInstance material,
                                bool overrideInherited )
  {
    if ( rb == nullptr )
      return;

    updateRtMaterial( getOrCreateRtData( rb ),
                      material,
                      overrideInherited );

    for ( const auto& geometry : rb->getGeometries() ) {
      // Only do this if the geometry has an inherited material because
      // we don't want to assign new material to shapes of an owning
      // parent geometry. E.g.,
      //   updateRtSurfaceMaterialHandle( rb->getGeometries()[47], ... );
      //   updateRtSurfaceMaterialHandle( rb, ... );
      // it's unexpected the shapes of geometry index 47 has different
      // material from their parent geometry.
      if ( !isRtMaterialOwner( geometry ) )
        updateRtMaterial( geometry, material, false );
    }
  }

  template<typename T>
  inline RtMaterialInstance getRtMaterial( const T& instance )
  {
    if ( instance == nullptr )
      return {};

    const RtInternalData* rtData = getRtData( instance );
    return rtData != nullptr ?
             rtData->getMaterial() :
             RtMaterialInstance{};
  }

  template<typename T>
  RtEntityId getOrCreateRtEntityIdImpl( T&& instance,
                                        RtEntityId hint = NullRtEntityId,
                                        bool overrideInherited = false )
  {
    if ( instance == nullptr )
      return NullRtEntityId;

    RtInternalData* rtData = getOrCreateRtData( instance );

    // Preventing 'create' when this instance already has its own id.
    overrideInherited = overrideInherited &&
                        !rtData->state.flags.Is( RtInternalData::State::ENTITY_ID_OWNER );

    // If, e.g., a geometry got overridden, its non-owning shapes should
    // receive the new geometry id 'hint'.
    // NOTE: Is this a bad (unexpected) side effect, remove it and make
    //       it explicit as an argument for the caller do deside.
    const auto updateNonOwningId = !overrideInherited &&
                                    hint != NullRtEntityId &&
                                    hint != rtData->getEntityId()  &&
                                    !rtData->state.flags.Is( RtInternalData::State::ENTITY_ID_OWNER );

    if ( rtData->getEntityId() == NullRtEntityId || overrideInherited || updateNonOwningId ) {
      // Assuming 'hint' not null is because there's a single other owner
      // of this id. E.g., rigid body:
      // auto rbId = getOrCreateRtEntityId( rb ); <- hint == null == owner
      //   for ( auto& geometry : rb->getGeometries() )
      //     getorCreateRtEntityId( geometry, rbId ); <- hint != null == NOT owner
      const auto isOwner = hint == NullRtEntityId || overrideInherited;
      rtData->state.flags.update( RtInternalData::State::ENTITY_ID_OWNER,
                                  isOwner );

      rtData->setEntityId( isOwner ?
                              RtRegistry::createId() :
                              hint );
    }

    return rtData->getEntityId();
  }

  template<typename T>
  RtEntityId getOrCreateRtEntityId( T&& instance,
                                    RtEntityId hint = NullRtEntityId,
                                    bool overrideInherited = false )
  {
    if ( instance == nullptr )
      return NullRtEntityId;
    else if ( auto track = instance->template asSafe<agxVehicle::Track>() )
      return getOrCreateRtEntityId( track, hint, overrideInherited );

    return getOrCreateRtEntityIdImpl( instance, hint, overrideInherited );
  }

  AGXSENSOR_EXPORT RtEntityId getOrCreateRtEntityId( agxVehicle::Track* track,
                                                     RtEntityId hint = NullRtEntityId,
                                                     bool overrideInherited = false );

  template<typename T>
  RtEntityId getRtEntityId( const T& instance )
  {
    const RtInternalData* rtData = getRtData( instance );
    return rtData != nullptr ?
             rtData->getEntityId() :
             NullRtEntityId;
  }

  template<typename T>
  bool isRtEntityIdOwner( const T& instance )
  {
    const RtInternalData* rtData = getRtData( instance );
    return rtData != nullptr &&
           rtData->state.flags.Is( RtInternalData::State::ENTITY_ID_OWNER );
  }

  template<typename T>
  bool shouldOverrideRtEntityId( const T& instance,
                                 RtEntityId currentId,
                                 bool overrideInherited )
  {
    // Override inherited if desired...:
    return overrideInherited &&
           // ...and instance already has an id...
           currentId != NullRtEntityId &&
           // ...that is inherited (not owner).
           !isRtEntityIdOwner( &instance );
  }

  /**
  \return the current id of the shape or a new id if overrideInherited = true and
          the shape wasn't owning its previous id
  */
  inline RtEntityId manageRtEntityId( agxCollide::Shape* shape,
                                      bool overrideInherited = true )
  {
    return getOrCreateRtEntityId( shape,
                                  NullRtEntityId,
                                  overrideInherited );
  }

  /**
  \return the current id of the geometry or a new id if overrideInherited = true and
          the geometry wasn't owning its previous id. If a new id is created, the new
          id is propagated to all shapes that doesn't already own its id.
  */
  inline RtEntityId manageRtEntityId( agxCollide::Geometry* geometry,
                                      bool overrideInherited = true )
  {
    if ( geometry == nullptr )
      return NullRtEntityId;

    auto geometryId = getRtEntityId( geometry );

    const auto doOverride = shouldOverrideRtEntityId( *geometry,
                                                      geometryId,
                                                      overrideInherited );
    if ( geometryId == NullRtEntityId || doOverride ) {
      geometryId = getOrCreateRtEntityId( geometry,
                                          geometryId,
                                          doOverride );

      // Propagating id of the geometry to the shapes if the shape
      // doesn't already have its own.
      for ( const auto& shape : geometry->getShapes() )
        getOrCreateRtEntityId( shape,
                               geometryId,
                               false );
    }

    return geometryId;
  }

  /**
  \return the current id of the rigid body or a new id if overrideInherited = true and
          the rigid body wasn't owning its previous id. If a new id is created, the new
          id is propagated to all geometries (and shapes) that doesn't already own its id.
  */
  inline RtEntityId manageRtEntityId( agx::RigidBody* rb,
                                      bool overrideInherited = true )
  {
    if ( rb == nullptr )
      return NullRtEntityId;

    auto rbId = getRtEntityId( rb );

    const auto doOverride = shouldOverrideRtEntityId( *rb,
                                                      rbId,
                                                      overrideInherited );
    if ( rbId == NullRtEntityId || doOverride ) {
      rbId = getOrCreateRtEntityId( rb,
                                    rbId,
                                    doOverride );

      // Propagating id if the rigid body to all its geometries and shapes.
      for ( const auto& geometry : rb->getGeometries() ) {
        getOrCreateRtEntityId( geometry,
                               rbId,
                               false );

        // Stop the propagation to shapes if this geometry is
        // id-owner since the shapes of it may be non-owners and
        // it's expected they have the id of the geometry.
        if ( isRtEntityIdOwner( geometry ) )
          continue;

        for ( const auto& shape : geometry->getShapes() )
          getOrCreateRtEntityId( shape,
                                 rbId,
                                 false );
      }
    }

    return rbId;
  }

  DOXYGEN_END_INTERNAL_BLOCK()
}
