
/*
Copyright 2007-2025. Algoryx Simulation AB.
All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.
Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <memory>
#include <string>
#include <unordered_set>
#include <unordered_map>
#include <openplx/Error.h>
#include <agxSDK/Assembly.h>
#include <agxPowerLine/PowerLine.h>
#include <agxPowerLine/Unit.h>
#include <agxPowerLine/RotationalUnit.h>
#include <agxPowerLine/Actuator1DOF.h>
#include <agxPowerLine/Connector.h>
#include <agxDriveTrain/Shaft.h>
#include <agxDriveTrain/Engine.h>
#include <agxDriveTrain/Gear.h>
#include <agxDriveTrain/GearBox.h>
#include <agxDriveTrain/Differential.h>
#include <agxDriveTrain/TorqueConverter.h>
#include <agxDriveTrain/DryClutch.h>
#include <agxDriveTrain/Brake.h>
#include <agxDriveTrain/CombustionEngine.h>
#include <agxOpenPLX/export.h>

namespace openplx {
    class ErrorReporter;
}

namespace openplx::Core {
    class Object;
}

namespace openplx::Physics {
    class System;
}

namespace openplx::Physics1D::Interactions {
    class Mate;
    class RotationalVelocityMotor;
}

namespace openplx::Physics1D::Bodies {
    class RotationalBody;
}

namespace openplx::DriveTrain {
    class Gear;
    class GearBox;
    class Differential;
    class EmpiricalTorqueConverter;
    class ManualClutch;
    class CombustionEngine;
    class ManualBrake;
}

namespace agxopenplx {

    enum class DriveTrainConstraintMapMode {
        Name,
        Uuid
    };

    /**
     * Maps the drivetrain components of an OpenPLX scene (Physics.System) into AGX powerline/drivetrain
     * components. Connects the drivetrain to the 3D actuators by looking them up from the assembly based
     * on either the OpenPLX name or the UUIDv5 mapped OpenPLX UUID.
     */
    class AGXOPENPLX_EXPORT OpenPlxDriveTrainMapper {
        public:
            OpenPlxDriveTrainMapper(
                std::shared_ptr<openplx::ErrorReporter> error_reporter,
                DriveTrainConstraintMapMode mode = DriveTrainConstraintMapMode::Uuid);

            void mapDriveTrainIntoPowerLine(
                const std::shared_ptr<openplx::Physics::System>& system,
                agxPowerLine::PowerLineRef power_line,
                agxSDK::AssemblyRef assembly
            );

            static int mapGearBoxGear(
                const std::shared_ptr<openplx::DriveTrain::GearBox>& gear_box,
                const agxDriveTrain::GearBoxRef& agx_gear_box,
                int gear);

        private:
            /**
             * Some mappings requre m_mate_actuator_set and m_drivetrain_unit_map to be populated.
             * We need to traverse the tree of Physics.Systems in passes to make sure the order is correct.
             * Pass1: Maps units and actuators
             * Pass2: Maps powerline interactions
             * Pass3: Maps assembly interactions
             */
            void mapSystemPass1(const std::shared_ptr<openplx::Physics::System>& system, agxPowerLine::PowerLineRef power_line);
            void mapSystemPass2(const std::shared_ptr<openplx::Physics::System>& system, agxPowerLine::PowerLineRef power_line);
            void mapSystemPass3(const std::shared_ptr<openplx::Physics::System>& system, agxSDK::AssemblyRef assembly);

            agxPowerLine::RotationalUnitRef mapRotationalBody(const std::shared_ptr<openplx::Physics1D::Bodies::RotationalBody>& body);
            agxDriveTrain::VelocityConstraintRef map1DRotationalVelocityMotor(const openplx::Physics1D::Interactions::RotationalVelocityMotor& motor);
            agxDriveTrain::GearRef mapGear(const std::shared_ptr<openplx::DriveTrain::Gear>& gear, const std::string& gear_key, const openplx::Physics::System& system);
            agxDriveTrain::GearBoxRef mapGearBox(const std::shared_ptr<openplx::DriveTrain::GearBox>& gear_box, const std::string& gear_box_key, const openplx::Physics::System& system);
            agxDriveTrain::DifferentialRef mapDifferential(const std::shared_ptr<openplx::DriveTrain::Differential>& differential, const std::string& differential_key, const openplx::Physics::System& system);
            agxDriveTrain::TorqueConverterRef mapTorqueConverter(const std::shared_ptr<openplx::DriveTrain::EmpiricalTorqueConverter>& torque_converter, const std::string& torque_converter_key, const openplx::Physics::System& system);
            agxDriveTrain::DryClutchRef mapClutch(const std::shared_ptr<openplx::DriveTrain::ManualClutch>& clutch, const std::string& clutch_key, const openplx::Physics::System& system);
            agxDriveTrain::BrakeRef mapBrake(const std::shared_ptr<openplx::DriveTrain::ManualBrake>& brake, const std::string& brake_key, const openplx::Physics::System& system);
            agxDriveTrain::CombustionEngineRef mapCombustionEngine(const std::shared_ptr<openplx::DriveTrain::CombustionEngine>& combustion_engine, const std::string& torque_converter_key, const openplx::Physics::System& system);
            void connectBetweenShafts(const std::shared_ptr<openplx::Physics1D::Interactions::Mate>& mate_1d, agxPowerLine::Connector* agx_connector, const std::string& mate_key, const openplx::Physics::System& system );
            void mapActuators(agxPowerLine::PowerLineRef power_line, agxSDK::AssemblyRef assembly);

            void reportErrorFromKey(const std::string& key, openplx::error_code_t error_code, const openplx::Physics::System& system);

            std::shared_ptr<openplx::ErrorReporter> m_error_reporter;
            DriveTrainConstraintMapMode m_mode;
            std::unordered_set<std::shared_ptr<openplx::Core::Object>> m_mate_actuator_set;
            std::unordered_map<std::shared_ptr<openplx::Core::Object>, agxPowerLine::UnitRef> m_drivetrain_unit_map;

            friend class OpenPlxToAgxMapper;
    };
}
