
/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <unordered_map>
#include <unordered_set>
#include <string>
#include <filesystem>
#include <cstdint>
#include <agxSDK/Assembly.h>
#include <agxSDK/Simulation.h>
#include <agx/RigidBody.h>
#include <agx/Hinge.h>
#include <agx/Prismatic.h>
#include <agx/LockJoint.h>
#include <agx/BallJoint.h>
#include <agx/CylindricalJoint.h>
#include <agx/SPDMatrix3x3.h>
#include <agx/Frame.h>
#include <agxCollide/Shape.h>
#include <agxCollide/Geometry.h>
#include <agx/Uuid.h>
#include <openplx/Document.h>
#include <openplx/ModelDeclaration.h>
#include <openplx/VarAssignment.h>
#include <openplx/Node.h>
#include <openplx/Token.h>
#include <agxOpenPLX/export.h>

namespace agxopenplx {

    /**
     * Converting meshes to .openplx is not straightforward. We have three different modes
     * to support different use cases.
     *
     * 1. OBJ_EXPORT: Exports meshes to .obj files which are referenced by
     *                Physics3D.Charges.ExternalTriMeshGeometry models in the converted .openplx document
     * 2. INLINE: Meshes are inlined using Physics3D.Charges.TriMesh which includes the
     *            vertice and indices data in the .openplx document
     * 3. CACHED: Mesh data is cached in the AgxCache and the .openplx document will contain
     *            an empty Physics3D.Charges.TriMeshGeometry with a .uuid annotation that
     *            can used to retreive the mesh instance from the cache.
     */
    enum class AgxToOpenPlxTrimeshMode {
        OBJ_EXPORT,
        INLINE,
        CACHED
    };

    enum class AgxToOpenPlxMaterialNamingRule {
        OPENPLX_NAME,
        AGX_NAME,
        AGX_UUID
    };

    class AGXOPENPLX_EXPORT EnsureUniqueId {
        public:
            std::string ensureUniqueId(const std::string& name, const agx::Uuid& uuid, std::unordered_set<std::string>& used_names);
            std::string lookupId(const agx::Uuid& x_uuid);
            void clear();
        private:
            std::unordered_map<std::string, std::string> m_uuid_to_name;
    };

    class AGXOPENPLX_EXPORT AgxToOpenPlxMapper {
        public:

            /**
             * Creates an AgxToOpenPlxMapper that uses .openplx representation of trimesh.
             * @warning Trimesh mode OBJ_EXPORT is not valid (use other constructor that takes
             *          export_folder and obj_relative path for OBJ_EXPORT)
             * @param precision The precision to use when converting floating point numbers to strings
             */
            AgxToOpenPlxMapper(
                agxSDK::Simulation* simulation,
                AgxToOpenPlxTrimeshMode trimesh_mode,
                uint32_t precision = 6,
                bool discard_rigid_body_positions = false,
                bool regenerate_shape_uuids = false,
                AgxToOpenPlxMaterialNamingRule material_naming_rule = AgxToOpenPlxMaterialNamingRule::AGX_NAME);

            /**
             * Creates an AgxToOpenPlxMapper that exports .obj files when mapping trimesh geometries,
             * will use TrimeshMode OBJ_EXPORT.
             * @param export_folder The folder where to put the exported .obj files
             * @param obj_relative_path Wether or not to use @-strings to reference exported .obj files
             *                          using relative paths (only works if the .obj file is exported
             *                          to the same folder as the .openplx file).
             * @param precision The precision to use when converting floating point numbers to strings
             */
            AgxToOpenPlxMapper(
                const std::string& export_folder,
                bool obj_relative_path,
                uint32_t precision = 6,
                bool discard_rigid_body_positions = false,
                bool regenerate_shape_uuids = false);

            std::string nameOrUuid(const agx::Name& name, const agx::Uuid& uuid);
            std::string nameWithIndexOrUuid(const agx::Name& name, size_t index, const agx::Uuid& uuid);
            /**
             * Converts an agx assembly to an OpenPLX document.
             * NOTE: Sub assemblies are not supported, only converts objects in the root assembly.
             */
            openplx::DocPtr assemblyToDocument(const agxSDK::AssemblyRef& assembly);
            /**
             * Converts an agx assembly to an OpenPLX string
             * NOTE: Sub assemblies are not supported, only converts objects in the root assembly.
             */
            std::string assemblyToOpenPlx(const agxSDK::AssemblyRef& assembly);

        private:

            void mapShapes(const agxCollide::GeometryRef& agx_geometry, const openplx::ModelDeclPtr& owner_md, const openplx::DocPtr& document, size_t& index);
            void mapRigidBody(const agx::RigidBodyRef& agx_rigid_body, const openplx::ModelDeclPtr& rigid_body_md, const openplx::DocPtr& document);
            std::shared_ptr<openplx::ModelDeclaration> mapGeometry(const std::string& geometry_identifier,
                                                                 const std::vector<openplx::Token>& geometry_type_identifier,
                                                                 const openplx::DocPtr& document,
                                                                 const openplx::ModelDeclPtr& rigid_body_md,
                                                                 const agxCollide::GeometryRef& agx_geometry,
                                                                 const agxCollide::ShapeRef& agx_shape,
                                                                 const agx::AffineMatrix4x4& geometry_transform,
                                                                 bool annotate_with_uuid,
                                                                 bool is_visual);
            std::shared_ptr<openplx::ModelDeclaration> mapContactGeometry(const std::string& geometry_identifier,
                                                                 const std::string& geometry_type_identifier,
                                                                 const openplx::DocPtr& document,
                                                                 const openplx::ModelDeclPtr& rigid_body_md,
                                                                 const agxCollide::GeometryRef& agx_geometry,
                                                                 const agxCollide::ShapeRef& agx_shape,
                                                                 const agx::AffineMatrix4x4& geometry_transform,
                                                                 bool annotate_with_uuid);
            std::shared_ptr<openplx::ModelDeclaration> mapVisualGeometry(const std::string& geometry_identifier,
                                                                 const std::string& geometry_type_identifier,
                                                                 const openplx::DocPtr& document,
                                                                 const openplx::ModelDeclPtr& rigid_body_md,
                                                                 const agxCollide::GeometryRef& agx_geometry,
                                                                 const agxCollide::ShapeRef& agx_shape,
                                                                 const agx::AffineMatrix4x4& geometry_transform,
                                                                 bool annotate_with_uuid);
            void insertAgxShapeIdentifier(const agxCollide::GeometryRef& agx_geometry, const agxCollide::ShapeRef& agx_shape, const std::string& geometry_identifier);
            void mapCachedShape(
                const agxCollide::ShapeRef& agx_shape,
                const agxCollide::GeometryRef& agx_geometry,
                const openplx::ModelDeclPtr& owner_md,
                const openplx::DocPtr& document,
                size_t index);
            void mapNonCachedShape(
                const agxCollide::ShapeRef& agx_shape,
                const agxCollide::GeometryRef& agx_geometry,
                const openplx::ModelDeclPtr& owner_md,
                const openplx::DocPtr& document,
                size_t index);
            void mapShape(const agxCollide::ShapeRef& agx_shape, const openplx::ModelDeclPtr& geometry_md, const openplx::DocPtr& document);
            void mapHinge(const agx::HingeRef& agx_hinge, const openplx::ModelDeclPtr& hinge_md, const openplx::DocPtr& document);
            void mapPrismatic(const agx::PrismaticRef& agx_prismatic, const openplx::ModelDeclPtr& prismatic_md, const openplx::DocPtr& document);
            void mapBall(const agx::BallJointRef& agx_ball_joint, const openplx::ModelDeclPtr& ball_md, const openplx::DocPtr& document);
            void mapLock(const agx::LockJointRef& agx_lock_joint, const openplx::ModelDeclPtr& lock_md, const openplx::DocPtr& document);
            void mapCylindrical(const agx::CylindricalJointRef& agx_cylindrical_joint, const openplx::ModelDeclPtr& cylindrical_md, const openplx::DocPtr& document);
            void mapMotor1D(
                const agx::Motor1D& motor_1d,
                const openplx::ModelDeclPtr& interaction_md,
                const openplx::VarAssignPtr& interaction_vd,
                const openplx::DocPtr& document,
                const std::string& motor_type,
                const std::string& name_suffix = "motor");
            void mapLock1D(
                const agx::Lock1D& lock_1d,
                const openplx::ModelDeclPtr& interaction_md,
                const openplx::VarAssignPtr& interaction_vd,
                const openplx::DocPtr& document,
                const std::string& position_param_name,
                const std::string& lock_type,
                const std::string& name_suffix = "spring");
            void mapRange1D(
                const agx::Range1D& range_1d,
                const openplx::ModelDeclPtr& interaction_md,
                const openplx::VarAssignPtr& interaction_vd,
                const openplx::DocPtr& document,
                const std::string& range_type,
                const std::string& name_suffix = "range");
            openplx::VarAssignPtr boolAssignment(const std::string& target, bool value);
            openplx::VarAssignPtr symbolAssignment(const std::string& target, std::string symbol);
            openplx::VarAssignPtr realAssignment(const std::string& target, double value);
            openplx::VarAssignPtr stringAssignment(const std::string& target, std::string value, bool at_string = false);
            openplx::VarAssignPtr realAssignmentIn(const std::vector<openplx::Token>& pre_target, const std::string& target, double value) const;
            openplx::Token realToken(double value) const;

            std::vector<openplx::VarAssignPtr> mapVelocities(const agx::RigidBodyRef& agx_rigid_body);
            std::vector<openplx::VarAssignPtr> mapFrame(const agx::FrameRef& agx_frame, const std::string& prefix = "local_transform");
            std::vector<openplx::VarAssignPtr> mapLocalTransform(const agx::AffineMatrix4x4& agx_affine_matrix);
            std::vector<openplx::VarAssignPtr> mapSPDMatrix3x3(const std::vector<openplx::Token>& target, const agx::SPDMatrix3x3& agx_matrix) const;

            template<class InteractionClassRef, class InteractionClass>
            openplx::VarAssignPtr mapInteraction(const InteractionClassRef& agx_interaction, const openplx::ModelDeclPtr& interaction_md, const openplx::DocPtr& document);

            template<class InteractionClassRef, class InteractionClass>
            void mapInteractionRegularization(
                const InteractionClassRef& agx_interaction,
                std::string dissipation_model_name,
                std::string flexibility_model_name,
                std::vector<std::pair<uint32_t, std::string>> dof_mapping,
                const openplx::ModelDeclPtr& interaction_md);

            openplx::MemberAccessPtr mapInteractionFrame(const std::string& interaction_name,
                                                       const agx::Frame& frame,
                                                       const agx::RigidBodyRef& rigid_body,
                                                       const std::vector<openplx::AnnotationPtr>& annotations = {});

            void mapCollisionGroups(const openplx::ModelDeclPtr& system_md);
            void mapMaterial(const agx::Material* agx_material, const openplx::DocPtr& document);
            void appendAssemblyFrameGeometryMap(const agxSDK::AssemblyRef& assembly, std::unordered_map<const agx::Frame*, agxCollide::Geometry*>& frame_geometry_map);
            void handleOrientedFrictionModel(std::vector<openplx::VarAssignPtr>& node_assignments, std::vector<openplx::TraitImplPtr>& friction_model_traits, const agx::ContactMaterial* agx_contact_material, const agx::Frame* reference_frame, agx::Vec3& primary_direction, const std::unordered_map<const agx::Frame*, agxCollide::Geometry*>& frame_geometry_map);
            bool appendReferenceGeometryIdentifierAndRotatePrimaryDirection(std::vector<openplx::VarAssignPtr> & node_assignments, const std::unordered_map<const agx::Frame*, agxCollide::Geometry*>& frame_geometry_map, const agx::Frame* frame, agx::Vec3& primary_direction);
            void mapContactMaterial(const agx::ContactMaterial* agx_contact_material, const openplx::ModelDeclPtr& system_md, const openplx::DocPtr& document, const std::unordered_map<const agx::Frame*, agxCollide::Geometry*>& frame_geometry_map);

            std::string uniqueModelName(const agx::Name& name, const agx::Uuid& uuid);

            agxSDK::Simulation* m_simulation;
            AgxToOpenPlxTrimeshMode m_trimesh_mode;
            bool m_obj_relative_path;
            std::filesystem::path m_export_folder;
            std::string m_root_system_identifier;
            std::unordered_map<const agx::RigidBody*, std::string> m_body_name_map;
            std::unordered_map<agxCollide::Shape*, std::string> m_shape_name_map;
            std::unordered_map<std::string, openplx::ModelDeclPtr> m_model_map;
            std::unordered_map<std::string, std::string> m_material_map;
            EnsureUniqueId m_ensure_unique_id;
            uint32_t m_precision;
            std::vector<openplx::Token> m_identifier_stack;
            std::multimap<agxCollide::GeometryRef, std::vector<openplx::Token>> m_geometry_segments;

            /* Settings */
            bool m_discard_rigid_body_positions;
            bool m_regenerate_shape_uuids;
            AgxToOpenPlxMaterialNamingRule m_material_naming_rule;
    };
}
