// Copyright 2022, Algoryx Simulation AB.

#pragma once

#include "CoreMinimal.h"

#include <memory>

struct FRigidBodyBarrier;
class FHingeBarrier;
class FSimulationBarrier;

/**
 * A collection of static helper functions used to create and manage the AGX Dynamics side of a
 * wheel loader drive-train.
 */
class WHEELLOADERBARRIER_API FDriveTrainUtilities
{
public:
	/**
	 * Opaque handle to the drive-train components that make up the wheel loader drive-train. Must
	 * be opaque because we are not allowed to name AGX Dynamics types in non-Barrier modules and
	 * this header file is included in the main Wheel Loader module, in DWL_DriveTrain.cpp.
	 * That module will only ever see pointers to this struct.
	 */
	struct FDriveTrainComponents;

	/*
	 * Handle to the drive-train components used by the wheel loader. This is how we pass state
	 * between the Unreal Engine layer in DWL_DriveTrain.cpp and the AGX Dynamics layer in
	 * DriveTrain.cpp.
	 *
	 * Must wrap the std::unique_ptr in a struct because only the Barrier module can know how to
	 * delete an FDriveTrainComponents instance, so the std::unique_ptr destructor code must live in
	 * a .cpp file in that module.
	 */
	struct WHEELLOADERBARRIER_API FDriveTrainId
	{
		std::unique_ptr<FDriveTrainComponents> Handle;

		// We don't have the definition for ~FDriveTrainComponent here so the compiler cannot
		// generate default implementations for these special member functions for us.
		FDriveTrainId();
		FDriveTrainId(std::unique_ptr<FDriveTrainComponents>&& InHandle);
		FDriveTrainId(FDriveTrainId&& Other);
		FDriveTrainId& operator=(FDriveTrainId&& Other) noexcept;
		~FDriveTrainId();
	};

	/// Create a wheel loader drive-train connected to the four given wheel hinges.
	static FDriveTrainId CreateDriveTrain(
		FSimulationBarrier& Simulation, FHingeBarrier* FrontLeftHinge,
		FHingeBarrier* FrontRightHinge, FHingeBarrier* RearLeftHinge,
		FHingeBarrier* RearRightHinge);

	static void SetGear(FDriveTrainId& Id, int32 Gear);
	static int32 GetGear(FDriveTrainId& Id);

	static void SetThrottle(FDriveTrainId& Id, float Throttle);
	static float GetThrottle(FDriveTrainId& Id);

	static float GetRpm(FDriveTrainId& Id);
};
