// Copyright 2022, Algoryx Simulation AB.

#pragma once

// Unreal Engine includes.
#include "CoreMinimal.h"
#include "Misc/AssertionMacros.h"

/*
 * The detect-functions are experimental for now. I'm trying to find a way to print a warning
 * message only once per Play instead of every frame without going through the Unreal Engine
 * macros since they send crash reports to Epic Games and it seems a bit wasteful to do that every
 * time the user has a typo in their name based references.
 */

void DetectTriggeredImpl(
	const ANSICHAR* Expr, const ANSICHAR* File, int32 Line, const TCHAR* Format, va_list Args)
{
	TCHAR MessageString[4096];
	FCString::GetVarArgs(MessageString, UE_ARRAY_COUNT(MessageString), Format, Args);

	TCHAR LocationAndExpressionString[MAX_SPRINTF];
	FCString::Sprintf(
		LocationAndExpressionString, TEXT("%s:%d: %s"), ANSI_TO_TCHAR(File), Line,
		ANSI_TO_TCHAR(Expr));
	UE_LOG(LogTemp, Error, TEXT("%s: %s"), LocationAndExpressionString, MessageString);
}

void DetectTriggered(
	const ANSICHAR* Expr, const ANSICHAR* File, int32 Line, const TCHAR* Format = TEXT(""), ...)
{
	va_list Args;
	va_start(Args, Format);
	DetectTriggeredImpl(Expr, File, Line, Format, Args);
	va_end(Args);
}

#define DWL_DETECT_IMPL(InExpression, bPrintedFlag, Format, ...)                        \
	(UNLIKELY(!!(InExpression)) && ([&]() {                                             \
		 if (!(*bPrintedFlag))                                                          \
		 {                                                                              \
			 (*bPrintedFlag) = true;                                                    \
			 DetectTriggered(#InExpression, __FILE__, __LINE__, Format, ##__VA_ARGS__); \
		 }                                                                              \
		 return true;                                                                   \
	 }()))

/**
 * Macro used to detect when an expression evaluates to true and print an error message once when
 * that happens. Intended to be used in `if` statements where the code block for the `if` handles
 * the error state.
 */
#define detect(InExpression, bPrintedFlag, InFormat, ...) \
	DWL_DETECT_IMPL(InExpression, bPrintedFlag, InFormat, ##__VA_ARGS__)

struct FAlwaysFalse
{
	// AlwaysFalse always evaluates for false in Boolean contexts.
	operator bool()
	{
		return false;
	}

	// Trying to make it true has no effect.
	FAlwaysFalse& operator=(bool)
	{
		return *this;
	}

	// AlwaysFalse imitates a pointer something that is always false.
	FAlwaysFalse& operator*()
	{
		return *this;
	}
} AlwaysFalse;
