// Copyright 2022, Algoryx Simulation AB.

#include "DriveTrain.h"

// Wheel Loader includes.
#include "Detect.h"

// AGX Dynamics includes.
#include "BeginAGXIncludes.h"
#include <agx/RigidBody.h>
#include <agx/Hinge.h>
#include <agx/version.h>
#include <agxDriveTrain/CombustionEngine.h>
#include <agxDriveTrain/Differential.h>
#include <agxDriveTrain/GearBox.h>
#include <agxDriveTrain/Shaft.h>
#include <agxDriveTrain/TorqueConverter.h>
#include <agxPowerLine/PowerLine.h>
#include <agxPowerLine/Actuator1DOF.h>
#include "EndAGXIncludes.h"

// AGX Dynamics for Unreal includes.
#include "AGX_AgxDynamicsObjectsAccess.h"
#include "AGX_Simulation.h"
#include "BarrierOnly/AGXTypeConversions.h"

// Unreal Engine includes.
#include "Misc/AssertionMacros.h"

FDriveTrainUtilities::FDriveTrainId::FDriveTrainId() = default;

FDriveTrainUtilities::FDriveTrainId::FDriveTrainId(
	std::unique_ptr<FDriveTrainComponents>&& InHandle)
	: Handle(std::move(InHandle))
{
}

FDriveTrainUtilities::FDriveTrainId::FDriveTrainId(FDriveTrainId&& Other) = default;

FDriveTrainUtilities::FDriveTrainId& FDriveTrainUtilities::FDriveTrainId::operator=(
	FDriveTrainId&& Other) noexcept = default;

FDriveTrainUtilities::FDriveTrainId::~FDriveTrainId() = default;

// FDriveTrainComponents holds all the AGX Dynamics drive-train components that we create.
struct FDriveTrainUtilities::FDriveTrainComponents
{
	agxPowerLine::PowerLineRef PowerLine;

	// These are the functional units within the drive-train, the components that perform some kind
	// of action on the torque it receives.
	agxDriveTrain::CombustionEngineRef Engine;
	agxDriveTrain::TorqueConverterRef TorqueConverter;
	agxDriveTrain::GearBoxRef GearBox;
	agxDriveTrain::DifferentialRef CenterDifferential;
	agxDriveTrain::DifferentialRef FrontDifferential;
	agxDriveTrain::DifferentialRef RearDifferential;
	agxPowerLine::RotationalActuatorRef FrontLeftActuator;
	agxPowerLine::RotationalActuatorRef FrontRightActuator;
	agxPowerLine::RotationalActuatorRef RearLeftActuator;
	agxPowerLine::RotationalActuatorRef RearRightActuator;

	// These are the coupling units within the drive-train, the components that link the functional
	// units together.
	agxDriveTrain::ShaftRef EngineToTorqueConverter;
	agxDriveTrain::ShaftRef TorqueConverterToGearBox;
	agxDriveTrain::ShaftRef GearBoxToCenterDifferential;
	agxDriveTrain::ShaftRef CenterDifferentialToFrontDifferential;
	agxDriveTrain::ShaftRef CenterDifferentialToRearDifferential;
	agxDriveTrain::ShaftRef FrontDifferentialToFrontLeftWheel;
	agxDriveTrain::ShaftRef FrontDifferentialToFrontRightWheel;
	agxDriveTrain::ShaftRef RearDifferentialToRearLeftWheel;
	agxDriveTrain::ShaftRef RearDifferentialToRearRightWheel;

	// There is currently no dedicated brake component in agxDriveTrain, so we mimic it with a hinge
	// and it's target speed controller on one of the coupling shafts.
	agx::HingeRef BrakeHinge;

	// Don't want to print the same error message every frame, so keep a flag for whether or not
	// each type of error message has been printed yet. These are reset on BeginPlay.
	bool bHavePrintedEngineError = false;
	bool bHavePrintedGearBoxError = false;
};

namespace DriveTrainUtilities_helpers
{
	// A collection of settings for the drive-train components that require configuration.
	// Consider making these Data Tables in Unreal Editor for easier configuration, at some point.
	struct FDriveTrainSettings
	{
		struct
		{
			agx::Real MaxTorque {1200.0};
			agx::Real MaxRPM {6000.0};
		} Engine;

		struct
		{
			agx::RealVector GearRatios {0.0, -10.0, 10.0};
		} GearBox;

		struct
		{
			agx::Real GearRatio {10.0};
			bool bLocked {true};
		} CenterDifferential;

		struct
		{
			agx::Real GearRatio {1.0};
			agx::Real SlipTorque {10000.0};
			bool bLocked {false};
		} FrontDifferential;

		struct
		{
			agx::Real GearRatio {1.0};
			agx::Real SlipTorque {10000.0};
			bool bLocked {false};
		} RearDifferential;

		struct
		{
			agx::Real PumpDiameter {0.11};
		} TorqueConverter;

		struct
		{
			agx::Real MaxBrakeTorque {1e6}; // No idea what this value should be.
		} Brake;
	};

	FDriveTrainUtilities::FDriveTrainId CreateDriveTrainImpl(
		agxSDK::Simulation& Simulation, agx::Hinge* FrontLeftHinge, agx::Hinge* FrontRightHinge,
		agx::Hinge* RearLeftHinge, agx::Hinge* RearRightHinge)
	{

		/*
		 Here is where we create and connect all the drive train components.
		 The layout is as follows:

		                                    Engine
		                                      |
		                                      |
		                                      v
		                               Torque converter
		                                      |
		                                      |
		                                      v
		    Front right wheel             Gear box                Rear right wheel
		             ^                        |                            ^
		             |                        | <-- Brakes                 |
		             |                        v                            |
		    Front differential <----- Center differential --------> Rear differential
		             |                                                     |
		             v                                                     v
		    Front left wheel                                         Rear left wheel


		*/

		FDriveTrainSettings Settings;

		// Create the scaffolding we need to expose the AGX Dynamics drive-train to the Unreal
		// Engine part of the project.
		FDriveTrainUtilities::FDriveTrainId DriveTrainId;
		DriveTrainId.Handle = std::make_unique<FDriveTrainUtilities::FDriveTrainComponents>();
		FDriveTrainUtilities::FDriveTrainComponents& DriveTrain = *DriveTrainId.Handle;

		// Create the drive-train container object.
		DriveTrain.PowerLine = new agxPowerLine::PowerLine();
		Simulation.add(DriveTrain.PowerLine);

		// Create and configure the engine based on the pre-defined Doosan-DE08TIS engine.
		// The engine is the source of torque in the drive-train. Throttle input from the player is
		// directed here.
		agxDriveTrain::CombustionEngineParameters EngineParams;
		agxDriveTrain::CombustionEngine::readLibraryProperties("Doosan-DE08TIS", EngineParams);

		// Changes to engine for a better driving experience.
		EngineParams.maxTorque = Settings.Engine.MaxTorque;
		EngineParams.maxRPM = Settings.Engine.MaxRPM;

		DriveTrain.Engine = new agxDriveTrain::CombustionEngine(EngineParams);
		DriveTrain.Engine->setEnable(true);
		DriveTrain.Engine->setThrottle(agx::Real(0.0));
		DriveTrain.PowerLine->add(DriveTrain.Engine);
		const agx::Real ShaftInertia = 0.075;

		// Create and configure the torque converter, which provide smoother torque transmission
		// out of the engine.
		DriveTrain.TorqueConverter = new agxDriveTrain::TorqueConverter();
		DriveTrain.TorqueConverter->setPumpDiameter(Settings.TorqueConverter.PumpDiameter);

		DriveTrain.PowerLine->add(DriveTrain.TorqueConverter);

		// Create and configure the gear box, which allows us to trade angular velocity for torque
		// at the wheels, and also to reverse the wheel turning direction.
		DriveTrain.GearBox = new agxDriveTrain::GearBox();
		DriveTrain.GearBox->setGearRatios(Settings.GearBox.GearRatios);
		DriveTrain.GearBox->gearUp();
		DriveTrain.PowerLine->add(DriveTrain.GearBox);

		// Create three differentials, which allow the wheels to turn independently and makes the
		// gear box - wheel coupling an equal-torque coupling instead of an equal-angular velocity
		// coupling.
		DriveTrain.CenterDifferential = new agxDriveTrain::Differential();
		DriveTrain.CenterDifferential->setGearRatio(Settings.CenterDifferential.GearRatio);
		DriveTrain.CenterDifferential->setLock(Settings.CenterDifferential.bLocked);
		DriveTrain.PowerLine->add(DriveTrain.CenterDifferential);

		DriveTrain.FrontDifferential = new agxDriveTrain::Differential();
		DriveTrain.FrontDifferential->setGearRatio(Settings.FrontDifferential.GearRatio);
		DriveTrain.FrontDifferential->setLock(Settings.FrontDifferential.bLocked);
		DriveTrain.FrontDifferential->setLimitedSlipTorque(Settings.FrontDifferential.SlipTorque);
		DriveTrain.PowerLine->add(DriveTrain.FrontDifferential);

		DriveTrain.RearDifferential = new agxDriveTrain::Differential();
		DriveTrain.RearDifferential->setGearRatio(Settings.RearDifferential.GearRatio);
		DriveTrain.RearDifferential->setLock(Settings.RearDifferential.bLocked);
		DriveTrain.RearDifferential->setLimitedSlipTorque(Settings.RearDifferential.SlipTorque);
		DriveTrain.PowerLine->add(DriveTrain.RearDifferential);

		// Create a rotational actuator for each wheel. These don't have any clear analogs in a
		// real-world drive train. In AGX Dynamics these are what bridges between the drive-train
		// components and the regular rigid bodies that make up the wheel loader. Each actuator
		// act on a constraint, hinges in this case, so we only create actuators for the hinges
		// we were able to find.
		if (FrontLeftHinge != nullptr)
		{
			DriveTrain.FrontLeftActuator = new agxPowerLine::RotationalActuator(FrontLeftHinge);
			DriveTrain.FrontLeftActuator->getInputShaft()->setInertia(ShaftInertia);
			detect(
				FrontLeftHinge->getMotor1D()->getEnable(), AlwaysFalse,
				TEXT("FrontLeftHinge has the speed controller enabled."));
			detect(
				FrontLeftHinge->getLock1D()->getEnable(), AlwaysFalse,
				TEXT("FrontLeftHinge has the lock controller enabled."));
		}
		if (FrontRightHinge != nullptr)
		{
			DriveTrain.FrontRightActuator = new agxPowerLine::RotationalActuator(FrontRightHinge);
			DriveTrain.FrontRightActuator->getInputShaft()->setInertia(ShaftInertia);
			detect(
				FrontRightHinge->getMotor1D()->getEnable(), AlwaysFalse,
				TEXT("FrontRightHinge has the speed controller enabled"));
			detect(
				FrontRightHinge->getLock1D()->getEnable(), AlwaysFalse,
				TEXT("FrontRightHinge has the lock controller enabled."));
		}
		if (RearLeftHinge != nullptr)
		{
			DriveTrain.RearLeftActuator = new agxPowerLine::RotationalActuator(RearLeftHinge);
			DriveTrain.RearLeftActuator->getInputShaft()->setInertia(ShaftInertia);
			detect(
				FrontRightHinge->getMotor1D()->getEnable(), AlwaysFalse,
				TEXT("RearLeftHinge has the speed controller enabled"));
			detect(
				RearLeftHinge->getLock1D()->getEnable(), AlwaysFalse,
				TEXT("RearLeftHinge has the lock controller enabled."));
		}
		if (RearRightHinge != nullptr)
		{
			DriveTrain.RearRightActuator = new agxPowerLine::RotationalActuator(RearRightHinge);
			DriveTrain.RearRightActuator->getInputShaft()->setInertia(ShaftInertia);
			detect(
				FrontRightHinge->getMotor1D()->getEnable(), AlwaysFalse,
				TEXT("RearRightHinge has the speed controller enabled"));
			detect(
				RearRightHinge->getLock1D()->getEnable(), AlwaysFalse,
				TEXT("RearRightHinge has the lock controller enabled."));
		}

		// Create and configure all the shafts we use to connect the functional units.
		DriveTrain.EngineToTorqueConverter = new agxDriveTrain::Shaft();
		DriveTrain.TorqueConverterToGearBox = new agxDriveTrain::Shaft();
		DriveTrain.GearBoxToCenterDifferential = new agxDriveTrain::Shaft();
		DriveTrain.CenterDifferentialToFrontDifferential = new agxDriveTrain::Shaft();
		DriveTrain.CenterDifferentialToRearDifferential = new agxDriveTrain::Shaft();
		DriveTrain.FrontDifferentialToFrontLeftWheel = new agxDriveTrain::Shaft();
		DriveTrain.FrontDifferentialToFrontRightWheel = new agxDriveTrain::Shaft();
		DriveTrain.RearDifferentialToRearLeftWheel = new agxDriveTrain::Shaft();
		DriveTrain.RearDifferentialToRearRightWheel = new agxDriveTrain::Shaft();

		DriveTrain.EngineToTorqueConverter->setInertia(ShaftInertia);
		DriveTrain.TorqueConverterToGearBox->setInertia(ShaftInertia);
		DriveTrain.GearBoxToCenterDifferential->setInertia(ShaftInertia);
		DriveTrain.CenterDifferentialToFrontDifferential->setInertia(ShaftInertia);
		DriveTrain.CenterDifferentialToRearDifferential->setInertia(ShaftInertia);
		DriveTrain.FrontDifferentialToFrontLeftWheel->setInertia(ShaftInertia);
		DriveTrain.FrontDifferentialToFrontRightWheel->setInertia(ShaftInertia);
		DriveTrain.RearDifferentialToRearLeftWheel->setInertia(ShaftInertia);
		DriveTrain.RearDifferentialToRearRightWheel->setInertia(ShaftInertia);

		DriveTrain.EngineToTorqueConverter->getRotationalDimension()->setName(
			"EngineToTorqueConverter");
		DriveTrain.TorqueConverterToGearBox->getRotationalDimension()->setName(
			"TorqueConverterToGearBox");
		DriveTrain.GearBoxToCenterDifferential->getRotationalDimension()->setName(
			"GearBoxToCenterDifferential");
		DriveTrain.CenterDifferentialToFrontDifferential->getRotationalDimension()->setName(
			"CenterDifferentialToFrontDifferential");
		DriveTrain.CenterDifferentialToRearDifferential->getRotationalDimension()->setName(
			"CenterDifferentialToRearDifferential");
		DriveTrain.FrontDifferentialToFrontLeftWheel->getRotationalDimension()->setName(
			"FrontDifferentialToFrontLeftWheel");
		DriveTrain.FrontDifferentialToFrontRightWheel->getRotationalDimension()->setName(
			"FrontDifferentialToFrontRightWheel");
		DriveTrain.RearDifferentialToRearLeftWheel->getRotationalDimension()->setName(
			"RearDifferentialToRearLeftWheel");
		DriveTrain.RearDifferentialToRearRightWheel->getRotationalDimension()->setName(
			"RearDifferentialToRearRightWheel");

		// Connect the components in the center column of the drive-train.
		//
		//                                  Engine
		//                                    |
		//                                    |
		//                                    v
		//                             Torque converter
		//                                    |
		//                                    |
		//                                    v
		//                                Gear box
		//                                    |
		//                                    |
		//                                    v
		//                            Center differential
		//
		DriveTrain.Engine->connect(DriveTrain.EngineToTorqueConverter);
		DriveTrain.EngineToTorqueConverter->connect(DriveTrain.TorqueConverter);
		DriveTrain.TorqueConverter->connect(DriveTrain.TorqueConverterToGearBox);
		DriveTrain.TorqueConverterToGearBox->connect(DriveTrain.GearBox);
		DriveTrain.GearBox->connect(DriveTrain.GearBoxToCenterDifferential);
		DriveTrain.GearBoxToCenterDifferential->connect(DriveTrain.CenterDifferential);

		// Connect the components in front part of the drive-train.
		//
		// Front right wheel
		//           ^
		//           |
		//           |
		//  Front differential <----- Center differential
		//           |
		//           v
		// Front left wheel
		//
		DriveTrain.CenterDifferential->connect(DriveTrain.CenterDifferentialToFrontDifferential);
		DriveTrain.CenterDifferentialToFrontDifferential->connect(DriveTrain.FrontDifferential);
		DriveTrain.FrontDifferential->connect(DriveTrain.FrontDifferentialToFrontLeftWheel);
		DriveTrain.FrontDifferential->connect(DriveTrain.FrontDifferentialToFrontRightWheel);
		if (DriveTrain.FrontLeftActuator != nullptr)
		{
			DriveTrain.FrontDifferentialToFrontLeftWheel->connect(DriveTrain.FrontLeftActuator);
		}
		if (DriveTrain.FrontRightActuator != nullptr)
		{
			DriveTrain.FrontDifferentialToFrontRightWheel->connect(DriveTrain.FrontRightActuator);
		}

		// Connect the components in the rear part of the drive-train.
		//
		//                                                          Rear right wheel
		//                                                                 ^
		//                                                                 |
		//                                                                 |
		//                            Center differential --------> Rear differential
		//                                                                 |
		//                                                                 v
		//                                                            Rear left wheel
		//
		DriveTrain.CenterDifferential->connect(DriveTrain.CenterDifferentialToRearDifferential);
		DriveTrain.CenterDifferentialToRearDifferential->connect(DriveTrain.RearDifferential);
		DriveTrain.RearDifferential->connect(DriveTrain.RearDifferentialToRearLeftWheel);
		DriveTrain.RearDifferential->connect(DriveTrain.RearDifferentialToRearRightWheel);
		if (DriveTrain.RearLeftActuator != nullptr)
		{
			DriveTrain.RearDifferentialToRearLeftWheel->connect(DriveTrain.RearLeftActuator);
		}
		if (DriveTrain.RearRightActuator != nullptr)
		{
			DriveTrain.RearDifferentialToRearRightWheel->connect(DriveTrain.RearRightActuator);
		}

		// Create the hinge that is used when braking. On real machines the brake is often applied
		// at the wheels, but for simplicity we break at the center column for now. This can be
		// improved in the future.
		//
		//                                Gear box
		//                                   |
		//                                   | <-- Brakes
		//                                   v
		//                            Center differential
		//
		agxPowerLine::RotationalDimensionRef ShaftDimension =
			DriveTrain.GearBoxToCenterDifferential->getRotationalDimension();
		ShaftDimension->reserveBody();
		agx::RigidBody* ShaftBody = ShaftDimension->getReservedBody();
		agx::Vec3 ShaftPosition = ShaftBody->getPosition();
		agx::Vec3 ShaftAxis = ShaftDimension->getWorldDirection();
		DriveTrain.BrakeHinge =
			agx::Constraint::createFromWorld<agx::Hinge>(ShaftPosition, ShaftAxis, ShaftBody);
		DriveTrain.BrakeHinge->getMotor1D()->setEnable(false);
		DriveTrain.BrakeHinge->getMotor1D()->setSpeed(agx::Real(0.0));
		DriveTrain.BrakeHinge->getMotor1D()->setForceRange(
			agx::RangeReal(Settings.Brake.MaxBrakeTorque));
		Simulation.add(DriveTrain.BrakeHinge);

		return DriveTrainId;
	}
}

FDriveTrainUtilities::FDriveTrainId FDriveTrainUtilities::CreateDriveTrain(
	FSimulationBarrier& Simulation, FHingeBarrier* FrontLeftHinge, FHingeBarrier* FrontRightHinge,
	FHingeBarrier* RearLeftHinge, FHingeBarrier* RearRightHinge)
{
	using namespace DriveTrainUtilities_helpers;

	agxSDK::Simulation* SimulationAgx = FAGX_AgxDynamicsObjectsAccess::GetFrom(&Simulation);
	if (SimulationAgx == nullptr)
	{
		UE_LOG(
			LogTemp, Warning,
			TEXT(
				"FDriveTrainUtilities::CreateDriveTrain got nullptr simulation. Cannot continue."));
		return {nullptr};
	}

	agx::Hinge* FrontLeftHingeAgx = FAGX_AgxDynamicsObjectsAccess::GetFrom(FrontLeftHinge);
	agx::Hinge* FrontRightHingeAgx = FAGX_AgxDynamicsObjectsAccess::GetFrom(FrontRightHinge);
	agx::Hinge* RearLeftHingeAgx = FAGX_AgxDynamicsObjectsAccess::GetFrom(RearLeftHinge);
	agx::Hinge* RearRightHingeAgx = FAGX_AgxDynamicsObjectsAccess::GetFrom(RearRightHinge);

	return CreateDriveTrainImpl(
		*SimulationAgx, FrontLeftHingeAgx, FrontRightHingeAgx, RearLeftHingeAgx, RearRightHingeAgx);
}

namespace FDriveTrainUtilities_helpers
{
	agxDriveTrain::GearBox* GetGearBox(
		FDriveTrainUtilities::FDriveTrainId& Id, const TCHAR* FunctionName)
	{
		static bool bNullDrivetrainErrorPrinted = false;
		if (detect(
				Id.Handle == nullptr, &bNullDrivetrainErrorPrinted,
				TEXT("Null DriveTrain passed to %s."), FunctionName))
		{
			return nullptr;
		}

		if (detect(
				Id.Handle->GearBox == nullptr, &Id.Handle->bHavePrintedGearBoxError,
				TEXT("Null GearBox passed to %s."), FunctionName))
		{
			return nullptr;
		}

		return Id.Handle->GearBox;
	}

	agxDriveTrain::CombustionEngine* GetEngine(
		FDriveTrainUtilities::FDriveTrainId& Id, const TCHAR* FunctionName)
	{
		static bool bNullDrivetrainErrorPrinted = false;
		if (detect(
				Id.Handle == nullptr, &bNullDrivetrainErrorPrinted,
				TEXT("Null DriveTrain passed to %s."), FunctionName))
		{
			return nullptr;
		}

		if (detect(
				Id.Handle->Engine == nullptr, &Id.Handle->bHavePrintedEngineError,
				TEXT("Null Engine passed to %s"), FunctionName))
		{
			return nullptr;
		}

		return Id.Handle->Engine;
	}
}

void FDriveTrainUtilities::SetGear(FDriveTrainId& Id, int32 Gear)
{
	using namespace FDriveTrainUtilities_helpers;
	agxDriveTrain::GearBox* GearBox = GetGearBox(Id, TEXT("SetGear"));
	if (GearBox == nullptr)
	{
		return;
	}

	size_t NumGears = GearBox->getNumGears();
	if (Gear < 0 || Gear >= NumGears)
	{
		UE_LOG(
			LogTemp, Warning, TEXT("Gear index %d is invalid for gear box with %d gears."), Gear,
			NumGears);
		return;
	}

	GearBox->setGear(Gear);
}

int32 FDriveTrainUtilities::GetGear(FDriveTrainId& Id)
{
	using namespace FDriveTrainUtilities_helpers;
	agxDriveTrain::GearBox* GearBox = GetGearBox(Id, TEXT("GetGear"));
	if (GearBox == nullptr)
	{
		return 0;
	}

	return GearBox->getGear();
}

void FDriveTrainUtilities::SetThrottle(FDriveTrainId& Id, float Throttle)
{
	using namespace FDriveTrainUtilities_helpers;
	agxDriveTrain::CombustionEngine* Engine = GetEngine(Id, TEXT("SetThrottle"));
	if (Engine == nullptr)
	{
		return;
	}

	Engine->setThrottle(Throttle);
}

float FDriveTrainUtilities::GetThrottle(FDriveTrainId& Id)
{
	using namespace FDriveTrainUtilities_helpers;
	agxDriveTrain::CombustionEngine* Engine = GetEngine(Id, TEXT("GetThrottle"));
	if (Engine == nullptr)
	{
		return 0.0f;
	}

	return ConvertToUnreal<float>(Engine->getThrottle());
}

float FDriveTrainUtilities::GetRpm(FDriveTrainId& Id)
{
	using namespace FDriveTrainUtilities_helpers;
	agxDriveTrain::CombustionEngine* Engine = GetEngine(Id, TEXT("GetRpm"));
	if (Engine == nullptr)
	{
		return 0.0f;
	}

	return ConvertToUnreal<float>(Engine->getRPM());
}

#undef detect
