// Copyright 2022, Algoryx Simulation AB.

#include "DWL_DriveTrainComponent.h"

// Wheel Loader includes.
#include "DriveTrain.h"

// AGX Dynamics for Unreal includes.
#include "AGX_Simulation.h"
#include "Constraints/AGX_HingeConstraintComponent.h"
#include "Constraints/HingeBarrier.h"

UDWL_DriveTrainComponent::UDWL_DriveTrainComponent()
{
	PrimaryComponentTick.bCanEverTick = false;

	FrontLeftWheelHinge.Name = TEXT("LeftFrontHinge");
	FrontRightWheelHinge.Name = TEXT("RightFrontHinge");
	RearLeftWheelHinge.Name = TEXT("LeftRearHinge");
	RearRightWheelHinge.Name = TEXT("RightRearHinge");
}

namespace DWL_helpers
{
	// Called when we know the final in-game owner of the given Reference.
	void SetComponentReferenceOwner(FAGX_SceneComponentReference& Reference, AActor* Owner)
	{
		if (Reference.OwningActor == nullptr)
		{
			Reference.OwningActor = Owner;
		}
	}

	FHingeBarrier* GetHingeBarrier(FAGX_SceneComponentReference& Reference)
	{
		USceneComponent* Component = Reference.GetSceneComponent();
		if (Component == nullptr)
		{
			UE_LOG(
				LogTemp, Warning,
				TEXT("UDWL_DriveTrain: Wheel reference '%s' does not point to a valid "
					 "ActorComponent. This wheel will be unpowered."),
				*Reference.Name.ToString())
			return nullptr;
		}

		UAGX_HingeConstraintComponent* HingeComponent =
			Cast<UAGX_HingeConstraintComponent>(Component);
		if (HingeComponent == nullptr)
		{
			UE_LOG(
				LogTemp, Warning,
				TEXT("UDWL_DriveTrain: Wheel hinge reference '%s' points to something not a hinge. "
					 "This wheel will be unpowered."),
				*Reference.Name.ToString());
			return nullptr;
		}

		FConstraintBarrier* ConstraintBarrier = HingeComponent->GetOrCreateNative();
		if (ConstraintBarrier == nullptr)
		{
			UE_LOG(
				LogTemp, Warning,
				TEXT("UDWL_DriveTrain: Could not get native AGX Dynamics hinge for reference "
					 "'%s'. This wheel will be unpowered."),
				*Reference.Name.ToString());
			return nullptr;
		}

		FHingeBarrier* HingeBarrier = HingeComponent->GetNativeHinge();
		if (HingeBarrier == nullptr)
		{
			UE_LOG(
				LogTemp, Warning,
				TEXT("UDWL_DriveTrain: Found a Hinge that contains a native AGX Dynamics "
					 "constraint of some type other than hinge in '%s'. This wheel will be "
					 "unpowered."),
				*Reference.Name.ToString());
		}

		return HingeBarrier;
	}

	FSimulationBarrier* GetSimulationBarrier(UActorComponent& Context)
	{
		UAGX_Simulation* SimulationSubsystem = UAGX_Simulation::GetFrom(&Context);
		if (SimulationSubsystem == nullptr)
		{
			UE_LOG(
				LogTemp, Warning,
				TEXT("UDWL_DriveTrain: Got nullptr SimulationSubsystem. Cannot continue."));
			return nullptr;
		}

		FSimulationBarrier* SimulationBarrier = SimulationSubsystem->GetNative();
		if (SimulationBarrier == nullptr)
		{
			UE_LOG(
				LogTemp, Warning,
				TEXT("UDWL_DriveTrain: Got nullptr SimulationBarrier. Cannot continue"));
			return nullptr;
		}

		return SimulationBarrier;
	}
}

void UDWL_DriveTrainComponent::SetGear(int32 Gear)
{
	FDriveTrainUtilities::SetGear(DriveTrainId, Gear);
}

int32 UDWL_DriveTrainComponent::GetGear()
{
	return FDriveTrainUtilities::GetGear(DriveTrainId);
}

void UDWL_DriveTrainComponent::SetThrottle(float Throttle)
{
	FDriveTrainUtilities::SetThrottle(DriveTrainId, Throttle);
}

float UDWL_DriveTrainComponent::GetThrottle()
{
	return FDriveTrainUtilities::GetThrottle(DriveTrainId);
}

float UDWL_DriveTrainComponent::GetRpm()
{
	return FDriveTrainUtilities::GetRpm(DriveTrainId);
}

void UDWL_DriveTrainComponent::BeginPlay()
{
	Super::BeginPlay();

	using namespace DWL_helpers;

	SetComponentReferenceOwner(FrontLeftWheelHinge, GetOwner());
	SetComponentReferenceOwner(FrontRightWheelHinge, GetOwner());
	SetComponentReferenceOwner(RearLeftWheelHinge, GetOwner());
	SetComponentReferenceOwner(RearRightWheelHinge, GetOwner());

	FSimulationBarrier* Simulation = GetSimulationBarrier(*this);
	if (Simulation == nullptr)
	{
		return;
	}

	FHingeBarrier* FrontLeftHinge = GetHingeBarrier(FrontLeftWheelHinge);
	FHingeBarrier* FrontRightHinge = GetHingeBarrier(FrontRightWheelHinge);
	FHingeBarrier* RearLeftHinge = GetHingeBarrier(RearLeftWheelHinge);
	FHingeBarrier* RearRightHinge = GetHingeBarrier(RearRightWheelHinge);

	DriveTrainId = FDriveTrainUtilities::CreateDriveTrain(
		*Simulation, FrontLeftHinge, FrontRightHinge, RearLeftHinge, RearRightHinge);
}
