# OpenPLX Tutorial 04 — Heavy Machine Bundle

This tutorial takes an ambitious step into creating a bundle tailored for modeling heavy machinery. A key vision of OpenPLX is to empower domain experts with domain-specific bundles that handle underlying physics modeling while providing a specialized ontology.

In this guide, we explore the technical process of creating a bundle and share design ideas for a domain-specific bundle dedicated to heavy machinery by describing the process of designing reusable crane modeling system.

The final template will in this tutorial be used to model a TBoom Crane without or with links. In later tutorials, this bundle will be extended to model an excavator arm.

<img src="images/crane_nolinks_thumbnail.png" alt="" width="500"/><img src="images/crane_links_thumbnail.png" alt="" width="500"/>

The image shows two crane configurations with and without links based off the same templates.

## Step 1: File structure

To create an OpenPLX bundle you create a `config.openplx` file in a folder.
It should have the following content:

```openplx
config is BundleConfig:
    name: "heavy_machine_bundle"
    dependencies: ["Physics3D", "Visuals", "Simulation"]
```

The entire content of the folder is now included in the bundle. Please note: nesting bundles within an OpenPLX bundle is not supported. All bundles must be defined in a completely flat structure. Within a bundle, you can create a folder structure which will define the namespaces for the models. Amongst other things, the config file makes it possible to reference openplx models between files and bundles.

Start by creating seven different folders to categorize different types of models:
- Connections
- Definitions
- Interfaces
- Linkages
- MachineTemplates
- MachineComponents
- Visuals

Also create a sub folder named `Traits` in both the `MachingComponent` and `Linkages` folders.
To understand which models we need, we will start with the requirements of building a TBoom crane, and then populate these folders with the identified connections, interfaces and machine components that we need. In later stages, we will be able to reuse the machine components to build other types of cranes.

---

## Step 2: Definitions and coordinates

To ensure compatibility between components and machines defined using the heavy machine bundle, we establish a set of vectors that define local normals and span the three dimensions. These vectors are defined in the `Definitions/orthonormal_basis.openplx` file as follows:

```
ModelPlaneNormal is Math.Vec3: Math.Vec3.Y_AXIS()
AlongBodyNormal is Math.Vec3: Math.Vec3.Z_AXIS()
ZeroAngleNormal is Math.Vec3: Definitions.ModelPlaneNormal.cross(Definitions.AlongBodyNormal)
```

* `ModelPlaneNormal`: Represents the local axle direction for components such as crane arms, excavator arms, or any structure with a natural 2D projection. By default, this vector points in the y-direction.
* `AlongBodyNormal`: Defines the natural length direction for components like booms or hydraulic cylinders. This vector points in the z-direction by default.
* `ZeroAngleNormal`: Establishes the zero direction for measuring angles around the `ModelPlaneNormal`. It is calculated as the cross product of `ModelPlaneNormal` and `AlongBodyNormal`.

## Step 3: Model the pivot TBoom Crane

The TBoom crane has one boom, which is connected to one stationary column, and is driven by a hydraulic cylinder.

We will start by focusing on an assembly of these three components without any additional linkages.

<img src="images/pivot-t-boom-stock.jpeg" alt="" width="500"/>

A typical TBoom loader crane.

### Boom Connection Interfaces
To connect the boom to the column, we start by defining the required interfaces. Both the column and the boom will have bearings at the connection point. To achieve this, we define an `AxleBearingInterface` in the `Interfaces/axle_bearing_interface.openplx` file. This interface includes three `internal_connection` connectors:

* `left_bearing` and `right_bearing`: Represent the bearings on either side of the connection
* `snap_center`: A central connector located between the bearings. This is used for assembling the crane in a central plane shared by all crane components.

The definition looks as follows:

```openplx
AxleBearingInterface:
    left_bearing is Physics3D.Interactions.MateConnector
    snap_center is Physics3D.Interactions.MateConnector
    right_bearing is Physics3D.Interactions.MateConnector
```

The connection between two `AxleBearingInterface`'s is defined by a `AxleBearingConnection` that we define in `Connections/axle_bearing_connection.openplx.`

```openplx
AxleBearingConnection is Physics3D.System:
    local_transform.position.x: 0
    input is Interfaces.AxleBearingInterface
    output is Interfaces.AxleBearingInterface
    bearing_left is Connections.BearingMate:
        connectors: [output.left_bearing, input.left_bearing]
    bearing_right is Connections.BearingMate:
        connectors: [output.right_bearing, input.right_bearing]
    snap_hinge is Physics3D.Interactions.Hinge:
        # Disable the physics
        enabled: false
        connectors: [output.snap_center, input.snap_center]
    snap_range is Physics3D.Interactions.RotationalRange:
        # Disable the physics
        enabled: false
        connectors: snap_hinge.connectors
        start: -Math.PI
        end: Math.PI
```
* The connection references two `AxleBearingInterface` for which it declares two `BearingMate` (`BearingMate` will be defined later) and one `Hinge` with disabled physics.
* The `snap_hinge` is purely for the `Physics3D` SNAP functionality to be able to assemble the system.
* The `snap_range` constrains the valid solutions for SNAP to assemble.

When a model includes physics declarations like the `Hinge` and `RotationalRange`, it has to inherit from `Physics3D.System` for the interactions to be discovered by the owner of the model. When introducing the `System` we create one additional, undesired, degree of freedom for SNAP to handle. This is removed by specifying a local transform of the system.

<img src="images/bearings-and-snap-range.png" alt="" width="500"/>

The wire frame image illustrates the `AxleBearingConnection` with the two bearings, and the `snap_center` internal_connection connector in the middle with a thin slice for illustrating the valid `snap_range`.

Next, define the  `BearingMate` in `Connections/bearing_internal_connection.openplx`:
```openplx
# A Cylindrical internal_connection which is quite flexible around the non-axial directions
BearingMate is Physics3D.Interactions.Cylindrical:
    # Rely on hinges in the AxleBearingConnections for assembly with SNAP
    enable_snap: false
    # Just to be super clear, this Mate will solve the physics for the bearings
    enabled: true
    # A slight friction around the axles
    friction_rotational becomes Physics.Interactions.Dissipation.DryScaleBoxFriction:
        coefficient: 0.001
    # The bearings are stiff in translational direction
    # but quite rotationally flexible
    flexibility becomes Physics3D.Interactions.Flexibility.LinearElasticCylindricalFlexibility:
        default_stiffness: 1E10
        around_normal.stiffness: 1E5
        around_cross.stiffness: 1E5

    dissipation becomes Physics3D.Interactions.Dissipation.CylindricalMechanicalDamping:
        default_damping: flexibility.default_stiffness * 0.01
        around_normal.damping_constant: flexibility.around_normal.stiffness * 0.01
        around_cross.damping_constant: flexibility.around_cross.stiffness * 0.01

```

SNAP should ignore the bearing internal_connections. This is specified by:  `enable_snap: false`.
The `BearingMate` extends the `Cylindrical` internal connection by introducing rotational friction around the cylindrical axis. Additionally, it allows flexibility in the other two rotational degrees of freedom to simulate the resistance of bearings outside the cylindrical axis. For these two degrees of freedom, dissipation is defined as mechanical damping.

Define two other interfaces that we will be need for connecting cylinders to the crane parts and placing `MateConnectors` relative to `CraneComponents`:

The `RedirectedCylinderConnector` in `Interfaces/redirected_cylinder_connector.openplx`.
```openplx
RedirectedCylinderConnector is Physics3D.Interactions.RedirectedMateConnector:
    main_axis: Definitions.AlongBodyNormal
    normal: Definitions.ZeroAngleNormal
```

Also a `RedirectedCraneAxleConnector` in `Interfaces/redirected_crane_axle_connector.openplx`:
```openplx
RedirectedCraneAxleConnector is Physics3D.Interactions.RedirectedMateConnector:
    main_axis: Definitions.ModelPlaneNormal
    normal: Definitions.ZeroAngleNormal
```

The `main_axis` and `normal` have pre defined local directions, to comply with the convention defined in the `orthonormal_basis.openplx` file.

### Visuals

Let's define the `CraneBodyVisuals` in `MachineComponents/crane_body.openplx`, with a set of visuals that adjust to the placements of the interfaces.

```openplx
CraneBodyVisuals is Physics3D.Bodies.RigidBody:
    drive_axle_visual is Visuals.AxleVisual
    ground_axle_visual is Visuals.AxleVisual
    guide_axle_visual is Visuals.AxleVisual
    output_axle_visual is Visuals.AxleVisual

    # Visuals for bearings
    input_bearing_left_visual is Visuals.BearingVisual
    input_bearing_right_visual is Visuals.BearingVisual

    # A convex visual and contact geometry for the crane body
    # defined by the axle positions
    wide_offset is Math.Vec3: {0,0.2,0}
    narrow_offset is Math.Vec3: {0,0.1,0}
    visual is Visuals.Geometries.ConvexMesh
    convex is Physics3D.Geometries.ConvexMesh:
        vertices: visual.vertices
```
Define the `AxleVisual` in `Visuals/axle_visual.openplx`

```
AxleVisual is Visuals.Geometries.Cylinder:
    radius: 0.02
    height: 0.5

```
and the `BearingVisual` in `Visuals/bearing_visual.openplx`

```
BearingVisual is Visuals.Geometries.Cylinder:
    radius: 0.03
    height: 0.045
```

Note that there is an `AxleVisual` for each interface, except the input, where we instead visualize two `BearingVisual` and one `ConvexMesh` which is defined by the placements of the connector interfaces.


### The Hydraulic Cylinder
To model the boom connections later, we need the definition of a hydraulic cylinder. We model it as a two body composition with a motorized linear degree of freedom in between. Start by creating a `MachineComponents/Traits/cylinder.openplx` file to define the `CylinderPart` trait:

```openplx
# A template for the piston and rod for a hydraulic cylinder
trait CylinderPart:
    bottom_height is Real: length * 0.5
    bearing_separation is Real: 0.15
    length is Real: 1.0

    # Connector to the other CylinderBody
    cylinder_connector is Interfaces.RedirectedCylinderConnector:
        position: length * Definitions.AlongBodyNormal

    # A central connector for connecting the cylinder to bearings
    snap_connector is Interfaces.RedirectedCraneAxleConnector:
        position.x: 0
    # The left and right bearings
    bearing_left is Interfaces.RedirectedCraneAxleConnector:
        position: snap_connector.position - bearing_separation * 0.5 * Definitions.ModelPlaneNormal
    bearing_right is Interfaces.RedirectedCraneAxleConnector:
        position: snap_connector.position + bearing_separation * 0.5 * Definitions.ModelPlaneNormal

    # Group the connectors
    axle_bearing_interface is Interfaces.AxleBearingInterface:
        left_bearing: bearing_left
        snap_center: snap_connector
        right_bearing: bearing_right
```

In the another file, `MachineComponents/cylinder_part.openplx`, define the `CylinderPartNoVisuals` system using the trait:
```openplx
CylinderPartNoVisuals is Physics3D.System:
    with Traits.CylinderPart
    body is Physics3D.Bodies.RigidBody:
        kinematics.local_transform.position.x: 0
    snap_connector.redirected_parent: body
    bearing_left.redirected_parent: body
    bearing_right.redirected_parent: body
    cylinder_connector.redirected_parent: body
```

The three `RedirectedCraneAxleConnector` from the trait together define the `AxleBearingInterface`.

Define a `CylinderBodyVisuals` with both a `Cylinder` visual and contact geometry as well as two `BearingVisual`s. The visuals will adapt to the parameters of the cylinder body.

```openplx
CylinderBodyVisuals is Physics3D.Bodies.RigidBody:
    # Visuals for bearings
    bearing_left_visual is Visuals.BearingVisual
    bearing_right_visual is Visuals.BearingVisual
    visual is Visuals.Geometries.Cylinder:
        radius: geometry.radius
        height: geometry.height
        local_transform: geometry.local_transform
    geometry is Physics3D.Geometries.Cylinder:
        radius: 0.1
        local_transform.rotation: Math.Quat.from_to(Math.Vec3.Y_AXIS(), Definitions.AlongBodyNormal)
```

And finally, also in the `cylinder_part.openplx` fils, define the `CylinderPart` using the `CylinderBodyVisuals`.
```openplx
CylinderPart is CylinderPartNoVisuals:
    body becomes CylinderBodyVisuals:
        bearing_left_visual.local_transform.position: bearing_left.position
        bearing_right_visual.local_transform.position: bearing_right.position
        geometry.height: length
        geometry.local_transform.position: Definitions.AlongBodyNormal * bottom_height
```

View the default configuration of the `CylinderPart` with `agxViewer`:
```bash
agxViewer MachineComponents/cylinder_part.openplx -p
```
<img src="images/cylinder-body.png" alt="" width="500"/>

Now we are ready to define a `HydraulicCylinderNoVisuals` in `MachineComponents/hydraulic_cylinder.openplx`.

```openplx
HydraulicCylinderNoVisuals is Physics3D.System:
    max_length is Real: 1.2
    stroke_length is Real: 0.6
    piston_length is Real: max_length - stroke_length
    rod_length is Real: stroke_length
    motor is Physics3D.Interactions.LinearVelocityMotor:
        connectors: internal_connection.connectors
        target_speed: 0
    # Declare the two cylinder bodies
    piston is MachineComponents.CylinderPartNoVisuals:
        length: piston_length
        local_transform.position.x: 0
    rod is MachineComponents.CylinderPartNoVisuals:
        length: rod_length
        # For the AxleBearingInterface of the Rod and the Piston
        # to be on opposite sides of the mounted hydraulic cylinder
        # we flip the direction of the main and normal.
        cylinder_connector.position: {0,0,0}
        cylinder_connector.main_axis: -piston.cylinder_connector.main_axis
        cylinder_connector.normal: -piston.cylinder_connector.normal

    # Limits the cylinder to the stroke length
    range is Physics3D.Interactions.LinearRange:
        connectors: internal_connection.connectors
        start: 0.0
        end: stroke_length

    # The internal_connection between the piston and the rod
    # Choose a cylindrical and not a prismatic
    # so that it does not constrain internal rotation
    internal_connection is Physics3D.Interactions.Cylindrical:
        connectors: [piston.cylinder_connector, rod.cylinder_connector]
        initial_angle: 0

    # Disable internal collisions
    cylinder_group is Simulation.CollisionGroup:
        systems: [this]
    cylinder_geometry_pair is Simulation.DisableCollisionPair:
        group1: cylinder_group
        group2: cylinder_group
```

Internal collisions are disabled with a `CollisionGroup` that include all of the hydraulic cylinder.

Use the `CylinderPart` to extend the model with visuals:
```openplx
HydraulicCylinder is HydraulicCylinderNoVisuals:
    piston becomes MachineComponents.CylinderPart
    rod becomes MachineComponents.CylinderPart:
        body.geometry.radius: 0.06
```

The `internal_connection`, defined in `HydraulicCylinderNoVisuals`, is modeled as a `Cylindrical` mate which allow the two `CylinderPart`s to both move relative each other along and around the cylinder axis.
The `stroke_length` limit the `LinearRange` of the cylinder. The `max_length` define the length of the cylinder in it's most extended state. Note that the connectors array of both the `range` and the `motor` are referencing the connectors of the `internal_connection` making sure they act on the same set of connectors.

Let's look at the the default configuration of the `HydraulicCylinder` with `agxViewer`:
```bash
agxViewer MachineComponents/hydraulic_cylinder.openplx -p
```

> With `Alt + Drag`, move the  outer piston to view the inner rod. Then press e start the simulation and see them snap back in place due to the defined internal connection.

<img src="images/hydraulic-cylinder.png" alt="" width="500"/>

In the image above a wire frame image of the hydraulic cylinder at zero extension is shown. The rod with the smaller radius can be seen inside the piston.


### Boom Connection Template Structure
It is now time to define a `BoomConnectionTemplate` in `Connections/boom_connection_template.openplx`. Start by defining the main template with references to components that represent parts of the crane. The declaration for these components will be covered in later steps.
```openplx
BoomConnectionTemplate is Physics3D.System:
    # Common template for the alternatives

    # Force system origin at origin of parent
    local_transform.position.x: 0

    # Two crane bodies
    boom_1 is MachineComponents.CraneSystemComponentNoVisuals
    boom_2 is MachineComponents.CraneSystemComponentNoVisuals
    # One hydraulic cylinder
    cylinder is MachineComponents.HydraulicCylinder

    # The connection between the two booms
    boom_1_boom_2_connection is Connections.AxleBearingConnection:
        input: boom_2.input
        output: boom_1.output

    # The connection of the cylinder to ground of the first boom
    boom_1_cylinder_connection is Connections.AxleBearingConnection:
        input: cylinder.piston.axle_bearing_interface
        output: boom_1.ground

    # Lock in axial direction of the second boom and the cylinder
    # to the first boom
    boom_1_boom_2_lock is Connections.AxleLock:
        connectors: [boom_1.output_bearing_left, boom_2.input_bearing_left]
    boom_1_cylinder_lock is Connections.AxleLock:
        connectors: [boom_1.ground_bearing_left, cylinder.piston.bearing_left]
```

Key Elements Explained:
1. `local_transform.position.x: 0`: This ensures that SNAP does not apply transformations to the system. It anchors the template in a fixed position along the x-axis.

2. Attributes `boom_1`, `boom_2`, and `cylinder`:

    * `boom_1` and `boom_2` represent crane booms.
    * `cylinder` represents a hydraulic cylinder.

3. Connections:
    * `boom_1_boom_2_connection`: Connects `boom_1` and `boom_2` via an `AxleBearingConnection`.
    * `boom_1_cylinder_connection`: Connects the hydraulic cylinder to `boom_1` through an `AxleBearingConnection`.
4. Locks:
    * `boom_1_boom_2_lock` and `boom_1_cylinder_lock` are `AxleLock` components used to restrict movement along the axle direction, ensuring stability and preventing unintended sliding.


The `BearingMate` of the `AxleBearingConnection` model a `Cylindrical` `internal_connection`, allowing booms and cylinder to slide along the axle direction. To avoid the slide, we have to lock the components along the axle direction. This could be avoided by using a `Hinge` instead of a `Cylindrical`, but would also result in an over-constrained system.

Next, define the `AxleLock` used in template in a new `Connections/axle_lock.openplx` file.

```openplx
AxleLock is Physics3D.Interactions.LinearSpring:
    position: 0
```

### Extend the template
The template structure can be reused to model a variety of systems. To model the TBoom crane specifically, the complete declaration of the  pivot TBoom crane connection require an additional `AxleBearingConnection` between the hydraulic cylinder and the boom. Lets declare the `PivotBoomConnection` in the `Connections/pivot_boom_connection.openplx` file by extending the `BoomConnectionTemplate`.
```openplx
PivotBoomConnection is BoomConnectionTemplate:
    cylinder_boom_snap is Connections.AxleBearingConnection:
        input: boom_2.drive
        output: cylinder.rod.axle_bearing_interface
```

### Crane components
Next, define the crane component used in the template as a `Physics3D.System`. When designing a simulation model with a blueprint or CAD model as source, you should match the coordinate system of the source with the one of the Physics3D.System.

A crane component need the required interfaces to be connected using i.e. the `PivotBoomConnection`.

Instead of immediately defining a crane component with all the necessary attributes, use the `OpenPLX Trait` feature to create a trait called `CraneSystemComponentTrait.` The trait feature is documented [here](https://openplx.org/learn/syntax/traits/).

>A `Trait` is a collection of attributes that can be reused by other types of OpenPLX models independent of the model type. By using traits, we can create models with similar structure and attributes without necessarily requiring one model to extend another.

Now, you can use the trait for any Physics3D system, and turn that system in to a crane component without needing to inherit from a specific type. The trait declares the shared attributes of the column and boom, focusing on functionality while deferring visuals. This allows flexibility in choosing the type of visualization later, whether using a simple representation or a detailed external tri-mesh.

We will create a definition for the heavy machine tutorial for crane system components according to this sketch:

<img src="images/crane-component.drawio.png" alt="" width="500"/>

The intention is that crane component should be possible to connect in a chain, where the actuators that actuate the next component are mounted on the previous component.
See the images of the cranes in the introduction to this tutorial.

A crane system component is designed with five potential connection points, each implemented as an AxleBearingInterface. These connections enable the modular and sequential construction of crane systems, where hydraulic cylinder actuators grounded in one crane body can actuate the next.

The naming convention for the five AxleBearingInterface's of a crane system component:

1. `drive` Interface:
    * Purpose: The primary point where the actuator exerts force or motion.
    * Role: Directly drives the movement of the crane body.

2. `guide` Interface:
    * Purpose: Connects guide links or other auxiliary components used in linked actuation.
    * Role: Helps maintain alignment and stability during operation, especially in systems with multiple linked components.

3. `ground` Interface:
    * Purpose: Provides the mounting point for hydraulic cylinders.
    * Role: Acts as the stable base for actuators, enabling controlled movement of other connected components.

4. `input` Interface:
    * Purpose: Connects to the output interface of the previous crane component in the series.
    * Role: Facilitates the flow of motion or control signals between sequential crane bodies.

5. `output` Interface:
    * Purpose: Serves as the connection point for the next crane component in the sequence.
    * Role: Propagates the actuation or movement to subsequent parts of the crane.


Create the `MachineComponents/Traits/crane_component.openplx` file and implement the `CraneComponent` trait.

```openplx
trait CraneComponent:
    # Default length of the crane body
    length is Real: 4
    # Local Z positions along the body for axles
    drive_along is Real: -length * 0.35 * 0.5
    ground_along is Real: length * 0.2 * 0.5
    guide_along is Real: length * 0.65 * 0.5
    drive_out is Real: 0.26
    ground_out is Real: 0.25
    guide_out is Real: 0.14

    # Default values for bearing and link separation
    drive_separation is Real: 0.2
    guide_separation is Real: 0.30
    ground_separation is Real: 0.2
    input_separation is Real: 0.2
    output_separation is Real: 0.2

    local_along_normal is Math.Vec3: Definitions.AlongBodyNormal
    local_model_plane_normal is Math.Vec3: Definitions.ModelPlaneNormal
    local_zero_angle_normal is Math.Vec3: local_model_plane_normal.cross(local_along_normal)

    # Central connectors for snap
    snap_drive is Interfaces.RedirectedCraneAxleConnector:
        position: drive_out * local_zero_angle_normal + drive_along * local_along_normal
    snap_ground is Interfaces.RedirectedCraneAxleConnector:
        position: ground_out * local_zero_angle_normal + ground_along * local_along_normal
    snap_guide is Interfaces.RedirectedCraneAxleConnector:
        position: guide_out * local_zero_angle_normal + guide_along * local_along_normal
    snap_input is Interfaces.RedirectedCraneAxleConnector:
        position: -0.5 * length * local_along_normal
    snap_output is Interfaces.RedirectedCraneAxleConnector:
        position: 0.5 * length * local_along_normal

    # Bearing connectors
    input_bearing_left is Interfaces.RedirectedCraneAxleConnector:
        position: snap_input.position - input_separation*0.5 * local_model_plane_normal
        main_axis: snap_input.main_axis
        normal: snap_input.normal
    input_bearing_right is Interfaces.RedirectedCraneAxleConnector:
        position: snap_input.position + input_separation*0.5 * local_model_plane_normal
        main_axis: snap_input.main_axis
        normal: snap_input.normal
    output_bearing_left is Interfaces.RedirectedCraneAxleConnector:
        position: snap_output.position - output_separation*0.5 * local_model_plane_normal
        main_axis: snap_output.main_axis
        normal: snap_output.normal
    output_bearing_right is Interfaces.RedirectedCraneAxleConnector:
        position: snap_output.position + output_separation*0.5 * local_model_plane_normal
        main_axis: snap_output.main_axis
        normal: snap_output.normal
    ground_bearing_left is Interfaces.RedirectedCraneAxleConnector:
        position: snap_ground.position - ground_separation*0.5 * local_model_plane_normal
        main_axis: snap_ground.main_axis
        normal: snap_ground.normal
    ground_bearing_right is Interfaces.RedirectedCraneAxleConnector:
        position: snap_ground.position + ground_separation*0.5 * local_model_plane_normal
        main_axis: snap_ground.main_axis
        normal: snap_ground.normal
    guide_bearing_left is Interfaces.RedirectedCraneAxleConnector:
        position: snap_guide.position - guide_separation*0.5 * local_model_plane_normal
        main_axis: snap_guide.main_axis
        normal: snap_guide.normal
    guide_bearing_right is Interfaces.RedirectedCraneAxleConnector:
        position: snap_guide.position + guide_separation*0.5 * local_model_plane_normal
        main_axis: snap_guide.main_axis
        normal: snap_guide.normal
    drive_bearing_left is Interfaces.RedirectedCraneAxleConnector:
        position: snap_drive.position - drive_separation*0.5 * local_model_plane_normal
        main_axis: snap_drive.main_axis
        normal: snap_drive.normal
    drive_bearing_right is Interfaces.RedirectedCraneAxleConnector:
        position: snap_drive.position + drive_separation*0.5 * local_model_plane_normal
        main_axis: snap_drive.main_axis
        normal: snap_drive.normal

    # Group connectors into axle connector triplets
    drive is Interfaces.AxleBearingInterface:
        snap_center: snap_drive
        left_bearing: drive_bearing_left
        right_bearing: drive_bearing_right
    guide is Interfaces.AxleBearingInterface:
        snap_center: snap_guide
        left_bearing: guide_bearing_left
        right_bearing: guide_bearing_right
    ground is Interfaces.AxleBearingInterface:
        snap_center: snap_ground
        left_bearing: ground_bearing_left
        right_bearing: ground_bearing_right
    input is Interfaces.AxleBearingInterface:
        snap_center: snap_input
        left_bearing: input_bearing_left
        right_bearing: input_bearing_right
    output is Interfaces.AxleBearingInterface:
        snap_center: snap_output
        left_bearing: output_bearing_left
        right_bearing: output_bearing_right
```

The connectors are positioned using the normals previously defined in `Definitions/orhonormal_basis.openplx`. This ensures that the three `RedirectedCraneAxleConnector` for each interface align on a line in the model plane of the crane.

Follow up by declaring the `CraneSystemComponentNoVisuals` in the `MachineComponents/crane_system_component.openplx` file, using the above defined trait.
```openplx
CraneSystemComponentNoVisuals is Physics3D.System:
    with Traits.CraneComponent # The with keyword extends a model with a trait

    body is Physics3D.Bodies.RigidBody:
        kinematics.local_transform.position.x: 0

    snap_drive.redirected_parent: body
    snap_ground.redirected_parent: body
    snap_guide.redirected_parent: body
    snap_input.redirected_parent: body
    snap_output.redirected_parent: body

    input_bearing_left.redirected_parent: body
    input_bearing_right.redirected_parent: body
    output_bearing_left.redirected_parent: body
    output_bearing_right.redirected_parent: body
    ground_bearing_left.redirected_parent: body
    ground_bearing_right.redirected_parent: body
    guide_bearing_left.redirected_parent: body
    guide_bearing_right.redirected_parent: body
    drive_bearing_left.redirected_parent: body
    drive_bearing_right.redirected_parent: body
```

The system declares a reference to a RigidBody (body) which is then set as the `redirected_parent` for all of the `RedirectedCraneAxleConnector`s. (input_bearing_eft etc.) that were defined in the trait.
By defining the interfaces in the system we can use an external body, i.e.

### Visuals for the crane
The `CraneSystemComponentNoVisuals` defined above is useful both when using static external tri-mesh for visualization, as well as when generating visuals from the crane geometry. In the prior case the trimesh will be guiding for where the axles and connections are positioned. But for virtual prototyping, an optimiztion of crane geometry, the visual and collision mesh need to adapt with updated axle positions and component size. For the later case, we define a crane system component with visuals that will adapt to the geometry and axle positions.

Extend the model above and declare the `CraneSystemComponent` where the interface data is fed to the visuals:
```openplx
CraneSystemComponent is CraneSystemComponentNoVisuals:
    body becomes CraneBodyVisuals
    # Visuals for axles
    body.drive_axle_visual.local_transform.position: snap_drive.position
    body.ground_axle_visual.local_transform.position: snap_ground.position
    body.guide_axle_visual.local_transform.position: snap_guide.position
    body.output_axle_visual.local_transform.position: snap_output.position

    # Visuals for bearings
    body.input_bearing_left_visual.local_transform.position: input_bearing_left.position
    body.input_bearing_right_visual.local_transform.position: input_bearing_right.position

    body.visual.vertices: [snap_drive.position -body.narrow_offset, snap_ground.position -body.narrow_offset, snap_guide.position -body.narrow_offset,
                   snap_input.position -body.wide_offset, snap_output.position -body.wide_offset, snap_drive.position +body.narrow_offset,
                   snap_ground.position +body.narrow_offset, snap_guide.position +body.narrow_offset, snap_input.position +body.wide_offset,
                   snap_output.position +body.wide_offset]
```

Let's view the default configuration of the `CraneSystemComponent` with `agxViewer`:
```bash
agxViewer MachineComponents/crane_system_component.openplx -p --modelName CraneSystemComponent
```
<img src="images/crane-body.png" alt="" width="500"/>

> By holding `i` and selecting the body in the viewer window, you can get information about the chosen body printed to the terminal.

Note! In tutorial [OpenPLX Tutorial 05 — Excavator](../t05-excavator/README.html) the `CraneSystemComponentNoVisuals` is used together with an external surce file with static geometry. See there how the `ExternalCraneSystemComponent` is defined.

### Assembling the pivot TBoom crane

Now we are ready to model the TBoom crane. Start by introducing a `TBoomCraneTemplateNoVisuals` model in `MachineTemplates/t_boom_crane_template.openplx` which we will use both for the pivot TBoom and later on for the linked TBoom crane.

```openplx
TBoomCraneTemplateNoVisuals is Physics3D.StructuralComponent:
    local_transform.position: -column.snap_input.position

    column is MachineComponents.CraneSystemComponentNoVisuals:
        local_transform.position.x: 0
        length: 2
        body.is_dynamic: false
        ground_along: -column.length * 0.6 * 0.5
    boom is MachineComponents.CraneSystemComponentNoVisuals:
        length: 2
    cylinder is MachineComponents.HydraulicCylinder:
        max_length: 2.0
        stroke_length: 1.0

    # Make the bearings align
    cylinder.piston.bearing_separation: column.ground_separation
    boom.input_separation: column.output_separation
    cylinder.rod.bearing_separation: boom.drive_separation

    boom_connection is Connections.BoomConnectionTemplate:
        boom_1: column
        boom_2: boom
        cylinder: cylinder
        # Make sure only positive angles are valid for connections
        # between the column.output to boom.input interfaces
        boom_1_boom_2_connection.snap_range.start: 0

    # Disable internal collisions
    no_collision_group is Simulation.CollisionGroup:
        systems: [cylinder, column, boom]

    geometry_pair is Simulation.DisableCollisionPair:
        group1: no_collision_group
        group2: no_collision_group
```
The template includes the column, boom and hydraulic cylinder as well as a `BoomConnectionTemplate` which declares the assembly of the parts.

The local position is set so that the input position of the column is at the crane origin. By specifying the column as the `reference_body` of the crane, SNAP will not update the local position of the column.

In the same file, follow this up with a `TBoomCraneTemplate` to enable visuals.

```openplx
TBoomCraneTemplate is TBoomCraneTemplateNoVisuals:
    column becomes MachineComponents.CraneSystemComponent
    boom becomes MachineComponents.CraneSystemComponent
```

View the template TBoom with agxViewer.
```bash
agxViewer MachineTemplates/t_boom_crane_template.openplx -p
```
<img src="images/t-boom-template.png" alt="" width="500"/>

Notice that the template TBoom has no specified connection between the cylinder and the boom, meaning the top part of the model will swing freely.

Now declare the `PivotTBoomCraneNoVisuals` in the `MachineTemplates/pivot_t_boom_crane.openplx file`, by extending the `TBoomCraneTemplateNoVisuals` and specializing its `boom_connection` to a `PivotBoomConnectionNoVisuals`.

```openplx
PivotTBoomCraneNoVisuals is TBoomCraneTemplateNoVisuals:
    boom_connection becomes Connections.PivotBoomConnection
```

In the same file we then declare the `PivotTBoomCrane` where all attributes becomes types with visuals.
```openplx
PivotTBoomCrane is PivotTBoomCraneNoVisuals:
    column becomes MachineComponents.CraneSystemComponent
    boom becomes MachineComponents.CraneSystemComponent
```

What might be hard to realize at this point is that the assembly of the model actually is ambiguous. We have two undefined parameters, but only one degree of freedom. We have to either specify the initial boom angle or the initial extent of the hydraulic cylinder.
It might be that we like to alter which data to specify, and therefore we create two variants of the crane
```openplx
PivotTBoomCraneDistance is PivotTBoomCrane:
    cylinder.internal_connection.initial_position: 0.3
```
which specifies the cylinder length, while the boom angle is computed by SNAP.

Load it with agxViewer.
```bash
agxViewer MachineTemplates/pivot_t_boom_crane.openplx -p --modelName PivotTBoomCraneDistance
```
<img src="images/pivot-t-boom-distance.png" alt="" width="500"/>

In the same file, create a variant of the pivot TBoom where the crane configuration is defined by the boom angle.

```openplx
PivotTBoomCraneAngle is PivotTBoomCrane:
    boom_connection.boom_1_boom_2_connection.snap_hinge.initial_angle: 1.2
```
The boom angle is specified in radians, and SNAP will compute the placement and extension of the cylinder.

```bash
agxViewer MachineTemplates/pivot_t_boom_crane.openplx -p --modelName PivotTBoomCraneAngle
```
<img src="images/pivot-t-boom-angle.png" alt="" width="500"/>

## Step 4: Model the linked TBoom Crane

The pivot connection plays a crucial role in determining the maximum load capacity of a loader crane. This capacity depends on the position of the drive interface relative to the boom input interface and the length of the boom.

Crane manufacturers often provide boom actuation systems that incorporate optimized linkages. These linkages are designed to adapt to specific load characteristics, ensuring efficient performance across various loading scenarios.



<img src="images/linked-system-stock.png" alt="Photograph of linked system in a TBoom Crane" width="500"/>
A common linked system for loader cranes

We will now show you how to model these links, and combine them with the existing crane template to create a linked crane.
### The link

Follow the same recipe as for the `CylinderPart` and the `CraneSystemComponent` and define the a `Link` trait in the `Linkages/Traits/link.openplx` file. The link can be used with any `Physics3D.System`.
```openplx
trait Link:
    length is Real: 0.1
    width is Real: 0.02
    height is Real: 0.05
    # The local position of the LinkConnector will offset the hole in local X
    input is Interfaces.RedirectedTelescopicLinkConnector
    output is Interfaces.RedirectedTelescopicLinkConnector
```
The input and output connectors are interfaces for connecting the link to other components.
Using the trait we define the `LinkNoVisuals` system in the `Linkages/link_no_visuals.openplx` file:
```openplx
LinkNoVisuals is Physics3D.System:
    with Traits.Link
    body is Physics3D.Bodies.RigidBody:
        kinematics.local_transform.position.x: 0
    input.redirected_parent: body
    output.redirected_parent: body
```
Any rigid body can now be assigned to the `body` attribute, which is placed at the origin of the system.

The characteristics of a linked connection might be something that we would like to optimize by trying a range of different lengths. To make this possible, declare a link that can be configurable in length, the `TelescopicLinkNoVisuals` in `Linkages/telescopic_link.openplx`:
```openplx
TelescopicLinkNoVisuals is LinkNoVisuals:
    length: 0.1
    width: 0.02
    height: 0.05
    # The local position of the LinkConnector will offset the hole in local X
    input becomes Interfaces.RedirectedTelescopicLinkConnector:
        local_position: -length * 0.5
    output becomes Interfaces.RedirectedTelescopicLinkConnector:
        local_position: length * 0.5
```
The link has two interfaces, the `input` and the `output` `LinkConnector`s, which are placed according to the `length` specified by the link.
Let the interfaces become the `RedirectedTelescopicLinkConnector` which can be adjusted along the link extension by passing the link `length` attribute.

Define the `RedirectedTelescopicLinkConnector` in `Interfaces/redirected_telescopic_link_connector.openplx`:

```openplx:
RedirectedTelescopicLinkConnector is RedirectedCraneAxleConnector:
    local_position is Real
    position: Definitions.AlongBodyNormal * local_position
```

The `TelescopicLinkNoVisuals` model might still be used with an external body. We like a body with an adaptive geometry and visual, that adjusts according to the dimensions of the link.
In the `Linkages/telescopic_link.openplx` file, define a body with visuals.
```openplx
TelescopicLinkBodyWithVisuals is Physics3D.Bodies.RigidBody:
    contact_geometry is Physics3D.Geometries.Box:
        enable_collisions: false
    visual_box is Visuals.Geometries.Box
    visual_cylinder_left is Visuals.BearingVisual
    visual_cylinder_right is Visuals.BearingVisual
```
The link body is represented by a visual box, and two cylinders symbolize the interfaces. Give it a `Box` contact geometry, but disable it by default to avoid collisions by default.

Use the body and declare extension with updated visuals in `TelescopicLink`:
```openplx
TelescopicLink is TelescopicLinkNoVisuals:
    body becomes TelescopicLinkBodyWithVisuals:
        kinematics.local_transform.position.x: 0
        contact_geometry.size: Definitions.AlongBodyNormal * (length + height) + Definitions.ModelPlaneNormal * width + Definitions.ZeroAngleNormal * height
        visual_box.size: body.contact_geometry.size
        visual_cylinder_left.local_transform.position: -length * 0.5 * Definitions.AlongBodyNormal
        visual_cylinder_right.local_transform.position: length * 0.5 * Definitions.AlongBodyNormal
```



View the link model:
```bash
agxViewer Linkages/telescopic_link.openplx -p
```

<img src="images/link.png" alt="" width="500"/>

### Paired Links

The links in cranes are mounted in pairs, with one right and one left item.
To simplify the modeling, declare a `LinkPairNoVisuals` component in the `Linkages/link_pair_no_visuals.openplx` file.

```openplx
LinkPairNoVisuals is Physics3D.System:
    length is Real: 0.4
    separation is Real: 0.1
    left is Linkages.LinkNoVisuals:
        local_transform.position: -separation * 0.5 * Definitions.ModelPlaneNormal

    right is Linkages.LinkNoVisuals:
        local_transform.position: separation * 0.5 * Definitions.ModelPlaneNormal
    # One extra central internal_connection connector for each hole
    snap_input is Interfaces.RedirectedCraneAxleConnector:
        redirected_parent: left.body
        position: left.input.position
    snap_output is Interfaces.RedirectedCraneAxleConnector:
        redirected_parent: left.body
        position: left.output.position

    input is Interfaces.AxleBearingInterface:
        left_bearing: left.input
        snap_center: snap_input
        right_bearing: right.input
    output is Interfaces.AxleBearingInterface:
        left_bearing: left.output
        snap_center: snap_output
        right_bearing: right.output
```
The two `AxleBearingInterface`s  allow us to connect another component to the link pair.


Create the `Linkages/telescopic_link_pair.openplx` and declare `TelescopicLinkPairNoVisuals` :
```openplx
TelescopicLinkPairNoVisuals is LinkPairNoVisuals:
    left becomes Linkages.TelescopicLinkNoVisuals:
        length: length
    right becomes Linkages.TelescopicLinkNoVisuals:
        length: length
```

Followed by
```openplx
TelescopicLinkPair is TelescopicLinkPairNoVisuals:
    left becomes Linkages.TelescopicLink
    right becomes Linkages.TelescopicLink
```

Load the link pair model (if no model name is defined in the command, the last model of the file will be loaded).
```bash
agxViewer Linkages/telescopic_link_pair.openplx -p
```

<img src="images/link-pair.png" alt="" width="500"/>

### The linked boom connection

Return to the `BoomConnectionTemplate` and extend it to a `LinkedBoomConnectionNoVisuals` model in `Connections/linked_boom_connection.openplx`.
```openplx
LinkedBoomConnectionNoVisuals is BoomConnectionTemplate:
    guide_links is Linkages.LinkPairNoVisuals:
        separation: boom_1.guide_separation
    drive_links is Linkages.LinkPairNoVisuals:
        separation: boom_2.drive_separation

    # Connect the different parts using AxleBearingConnection's
    guide_links_cylinder_connection is Connections.AxleBearingConnection:
        input: cylinder.rod.axle_bearing_interface
        output: guide_links.output
    drive_links_cylinder_connection is Connections.AxleBearingConnection:
        input: cylinder.rod.axle_bearing_interface
        output: drive_links.input
    guide_link_snap is Connections.AxleBearingConnection:
        input: boom_1.guide
        output: guide_links.input
    drive_link_snap is Connections.AxleBearingConnection:
        input: boom_2.drive
        output: drive_links.output


    # Snap assembles Mate by Mate
    # Snap will, as long as it is possible,
    # only move objects to already assembled objects.
    # If the number of Mates are too few to assemble everything,
    # snap will force some objects. If the number of Mates is too large,
    # there might be Mates that will not be possible to assemble,
    # however, the physics will still be enabled.
    # Where we have junctions, like where both link pairs connect
    # to the hydraulic cylinder, snap requires one Mate for each pair
    # The AxleBearingConnections above cover all of the physics,
    # but there is nothing that would snap the guide links to the
    # drive links. And the order of assembly could require that.
    # In the CraneAngle model below, the links need to meet first,
    # for the cylinder to afterward be pulled and rotated into place.
    # Therefore we need a snap_hinge for this.
    snap_hinge is Physics3D.Interactions.Hinge:
        enabled: false
        connectors: [guide_links.snap_output, drive_links.snap_input]

    # The snap_range may limit the valid range for the guide / drive links
    # to limit the possible solutions for the problem with two roots
    snap_range is Physics3D.Interactions.RotationalRange:
        # Disable the physics
        enabled: false
        connectors: snap_hinge.connectors
        start: -Math.PI
        end: Math.PI


    # Lock translational DOF of links to crane bodies
    guide_lock_left is Connections.AxleLock:
        connectors: [guide_links.left.input, boom_1.guide_bearing_left]
    guide_lock_right is Connections.AxleLock:
        connectors: [guide_links.right.input, boom_1.guide_bearing_right]
    drive_lock_left is Connections.AxleLock:
        connectors: [drive_links.left.output, boom_2.drive_bearing_left]
    drive_lock_right is Connections.AxleLock:
        connectors: [drive_links.right.output, boom_2.drive_bearing_right]
```
Here we introduce both the guide and drive link pairs. Instead of connecting the cylinder directly to the boom, we connect it to both link pairs by `guide_links_cylinder_connection` and `drive_links_cylinder_connection`. The `guide_links` are then connected by the `guide_link_snap` to the guide interface of the column while the `drive_links` are connected by the `drive_link_snap` to the drive interface of the boom. The link separation is set to the same distance as the distance between the assigned bearings of the crane bodies. On top of that, each link gets an `AxleLock` to constrain them to the crane bodies along the axle direction.

`Snap assembly` in OpenPLX builds the crane sequentially, attaching new components only to already assembled parts whenever possible. Since the guide links and drive links must align before the hydraulic cylinder connects, a `snap_hinge` is used to enforce this order, though it does not affect physics. The `snap_range` further restricts assembly positions to prevent ambiguity (i.e. the crane bending in the wrong direction). These snapping constraints serve as "hidden" background helpers that guide the correct assembly process, resulting in a stable and well-structured mechanism.

As previously, declare the `LinkedBoomConnection` with visuals in the same file:
```openplx
LinkedBoomConnection is LinkedBoomConnectionNoVisuals:
    guide_links becomes Linkages.TelescopicLinkPair
    drive_links becomes Linkages.TelescopicLinkPair
```

### Assembling the linked TBoom crane

Everything is in place to extend `TBoomCraneTemplate` to a `LinkedTBoomCrane`. Add the extended model to `MachineTemplates/linked_t_boom_crane.openplx`:
```
LinkedTBoomCrane is TBoomCraneTemplate:
    guide_links is Linkages.TelescopicLinkPair:
        separation: column.guide_separation
    drive_links is Linkages.TelescopicLinkPair:
        separation: boom.drive_separation

    boom_connection becomes Connections.LinkedBoomConnection:
        guide_links: guide_links
        drive_links: drive_links
        drive_link_snap.snap_range.start: 0
```
The linked pairs are declared, and the `boom_connection` is upgraded to a `LinkedBoomConnection`. Note the `drive_link_snap.snap_range.start: 0` declaration, which limit the available solution space for SNAP to rotate the drive links. This results in choosing the wanted solution. Try leaving this out, and you will get an invalid crane state.


Just as with the `PivotTBoomCrane`, declare two alternative models in the same file, the `LinkedTBoomCraneDistance` that specifies the crane position given the extension of the hydraulic cylinder, and the `LinkedTBoomCraneAngle` that depends on the boom angle.

```openplx
# A crane where the initial state is defined by the length of the cylinder
LinkedTBoomCraneDistance is LinkedTBoomCrane:
    cylinder.internal_connection.initial_position: 0.4

# A crane where the initial state is defined by the angle of the boom
LinkedTBoomCraneAngle is LinkedTBoomCrane:
    boom_connection.boom_1_boom_2_connection.hinge.initial_angle: 1.2
```

View the two alternatives:
```bash
agxViewer MachineTemplates/linked_t_boom_crane.openplx --modelName LinkedTBoomCraneDistance -p
```

```bash
agxViewer MachineTemplates/linked_t_boom_crane.openplx --modelName LinkedTBoomCraneAngle -p
```
>Try modifying the length or the angle and notice how the model changes.

<img src="images/linked-t-boom-distance.png" alt="" width="500"/><img src="images/linked-t-boom-angle.png" alt="" width="500"/>

Note the `drive_link_snap.snap_range.start: 0` declaration for the `boom_connection`, which limit the available solution space for SNAP to rotate the drive links. This results in choosing the wanted solution. Try leaving this out, and you will get an invalid crane state as in the images below.

<img src="images/invalid-linked-t-boom-distance.png" alt="" width="500"/><img src="images/invalid-linked-t-boom-angle.png" alt="" width="500"/>


## Different variants of the crane

We have now defined two templates for the TBoom crane, one with links and one without.
With our models declared, we can begin exploring the possibilities of OpenPLX.
Create a `t_boom_variants.openplx` file in the root folder of the bundle.

### Vary the link length
Now create two variants of the `LinkedTBoomCraneAngle`.
One with shorter guide links, and a modified hydraulic cylinder reach:
```openplx
LinkedTBoomCraneShortGuide is LinkedTBoomCraneAngle:
    guide_links.length: 0.3
    cylinder.stroke_length: 0.8
    cylinder.max_length: 1.8
```
Load `LinkedTBoomCraneShortGuide` with
```bash
agxViewer t_boom_variants.openplx --modelName LinkedTBoomCraneShortGuide
```

And one with longer guide links:
```openplx
LinkedTBoomCraneLongDrive is LinkedTBoomCraneAngle:
    drive_links.length: 0.5
```
Load `LinkedTBoomCraneLongDrive` with
```bash
agxViewer t_boom_variants.openplx --modelName LinkedTBoomCraneLongDrive
```

<img src="images/linked-t-boom-short-guide.png" alt="" width="500"/><img src="images/linked-t-boom-long-drive.png" alt="" width="500"/>

From the two images you can tell that the link-cylinder configuration differ for the two variants given the same boom angle.

### Alternate the interface placements

Now create two variants of the `PivotTBoomCraneDistance`.
One with a different placement of the drive interface of the boom:
```openplx
PivotTBoomCraneMoveDrive is MachineTemplates.PivotTBoomCraneDistance:
    boom.drive_along: -boom.length * 0.2 * 0.5
    boom.drive_out: 0.4
```
Load `PivotTBoomCraneMoveDrive` with
```bash
agxViewer t_boom_variants.openplx --modelName PivotTBoomCraneMoveDrive
```

And one with a different placement for the ground interface of the column:
```openplx
PivotTBoomCraneMoveGround is MachineTemplates.PivotTBoomCraneDistance:
    column.ground_along: column.length * 0.2 * 0.5
    column.ground_out: 0.3
```
Load `PivotTBoomCraneMoveGround` with
```bash
agxViewer t_boom_variants.openplx -p --modelName PivotTBoomCraneMoveGround
```

<img src="images/pivot-t-boom-move-drive.png" alt="" width="500"/><img src="images/pivot-t-boom-move-ground.png" alt="" width="500"/>

From the two images you can tell that the boom angle differ for the two variants given the same cylinder length.

## Conclusion
This tutorial has served as an introduction to OpenPLX Bundles. Your new Crane Bundle is designed to be flexible, modifiable and reusable to design many different configurations. In later tutorials, we will build upon this bundle to define an excavator arm using the same source models.
